<!-- Keyboard Font -->
<link rel="stylesheet" href="{{ asset('fonts/keyboard.css') }}">

<style>
    .keyboard {
        position: fixed;
        left: 0;
        bottom: 0;
        width: 100%;
        padding: 10px 0;
        background: #343A40;
        user-select: none;
        transition: bottom 0.4s;
        z-index: 9999;
    }

    .keyboard--hidden {
        bottom: -100%;
    }

    /* Each row is flex for professional alignment */
    .keyboard__row {
        display: flex;
        justify-content: center;
        margin-bottom: 5px;
    }

    .keyboard__key {
        height: 50px;
        width: 60px;
        margin: 3px;
        border-radius: 6px;
        border: none;
        background: rgba(255, 255, 255, 0.2);
        color: #ffffff;
        font-size: 1.1rem;
        outline: none;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        -webkit-tap-highlight-color: transparent;
        position: relative;
        transition: background 0.2s;
    }

    .keyboard__key:active {
        background: rgba(255, 255, 255, 0.12);
    }

    .keyboard__key--wide {
        width: 100px;
    }

    .keyboard__key--extra-wide {
        width: 300px;
    }

    .keyboard__key--dark {
        background: rgba(0, 0, 0, 0.3);
    }

    .keyboard__key--activatable::after {
        content: '';
        top: 10px;
        right: 10px;
        position: absolute;
        width: 8px;
        height: 8px;
        background: rgba(0, 0, 0, 0.4);
        border-radius: 50%;
    }

    .keyboard__key--active::after {
        background: #08ff00;
    }
</style>

<script>
    const Keyboard = {
        elements: {
            main: null,
            keysContainer: null,
            keys: []
        },

        eventHandlers: {
            oninput: null,
            onclose: null
        },

        properties: {
            value: "",
            capsLock: false
        },

        init() {
            this.elements.main = document.createElement("div");
            this.elements.main.classList.add("keyboard", "keyboard--hidden");

            // Build rows for professional layout
            const rows = [
                ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0", "backspace"],
                ["q", "w", "e", "r", "t", "y", "u", "i", "o", "p"],
                ["caps", "a", "s", "d", "f", "g", "h", "j", "k", "l", "enter"],
                ["done", "z", "x", "c", "v", "b", "n", "m", ",", ".", "?"],
                ["space"]
            ];

            rows.forEach(row => {
                const rowEl = document.createElement("div");
                rowEl.classList.add("keyboard__row");

                row.forEach(key => {
                    rowEl.appendChild(this._createKey(key));
                });

                this.elements.main.appendChild(rowEl);
            });

            document.body.appendChild(this.elements.main);

            document.querySelectorAll(".use-keyboard-input").forEach(element => {
                element.addEventListener("focus", () => {
                    this.open(element.value, currentValue => {
                        element.value = currentValue;
                        element.dispatchEvent(new Event('input'));
                    });
                });
            });
        },

        _createKey(key) {
            const keyElement = document.createElement("button");
            keyElement.setAttribute("type", "button");
            keyElement.classList.add("keyboard__key");

            const createIconHTML = (icon_name) => `<i class="material-icons">${icon_name}</i>`;

            switch (key) {
                case "backspace":
                    keyElement.classList.add("keyboard__key--wide");
                    keyElement.innerHTML = createIconHTML("backspace");
                    keyElement.addEventListener("click", () => {
                        this.properties.value = this.properties.value.slice(0, -1);
                        this._triggerEvent("oninput");
                    });
                    break;

                case "caps":
                    keyElement.classList.add("keyboard__key--wide", "keyboard__key--activatable");
                    keyElement.innerHTML = createIconHTML("keyboard_capslock");
                    keyElement.addEventListener("click", () => {
                        this._toggleCapsLock();
                        keyElement.classList.toggle("keyboard__key--active", this.properties.capsLock);
                    });
                    break;

                case "enter":
                    keyElement.classList.add("keyboard__key--wide");
                    keyElement.innerHTML = createIconHTML("keyboard_return");
                    keyElement.addEventListener("click", () => {
                        this.properties.value += "\n";
                        this._triggerEvent("oninput");
                    });
                    break;

                case "space":
                    keyElement.classList.add("keyboard__key--extra-wide");
                    keyElement.innerHTML = createIconHTML("space_bar");
                    keyElement.addEventListener("click", () => {
                        this.properties.value += " ";
                        this._triggerEvent("oninput");
                    });
                    break;

                case "done":
                    keyElement.classList.add("keyboard__key--wide", "keyboard__key--dark");
                    keyElement.innerHTML = createIconHTML("check_circle");
                    keyElement.addEventListener("click", () => {
                        this.close();
                        this._triggerEvent("onclose");
                    });
                    break;

                default:
                    keyElement.textContent = key.toLowerCase();
                    keyElement.addEventListener("click", () => {
                        this.properties.value += this.properties.capsLock ? key.toUpperCase() : key
                            .toLowerCase();
                        this._triggerEvent("oninput");
                    });
                    break;
            }
            return keyElement;
        },

        _triggerEvent(handlerName) {
            if (typeof this.eventHandlers[handlerName] === "function") {
                this.eventHandlers[handlerName](this.properties.value);
            }
        },

        _toggleCapsLock() {
            this.properties.capsLock = !this.properties.capsLock;
            for (const key of this.elements.main.querySelectorAll(".keyboard__key")) {
                if (key.childElementCount === 0) {
                    key.textContent = this.properties.capsLock ?
                        key.textContent.toUpperCase() :
                        key.textContent.toLowerCase();
                }
            }
        },

        open(initialValue, oninput, onclose) {
            this.properties.value = initialValue || "";
            this.eventHandlers.oninput = oninput;
            this.eventHandlers.onclose = onclose;
            this.elements.main.classList.remove("keyboard--hidden");
        },

        close() {
            this.properties.value = "";
            this.elements.main.classList.add("keyboard--hidden");
        }
    };

    window.addEventListener("DOMContentLoaded", () => Keyboard.init());
</script>
