<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class Sale extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'waiter_id',
        'date',
        'selling_id',
        'total',
        'discount',
        'net',
        'vat',
        'paid',
        'due',
        'account_id',
        'customer_id',
        'salesmen_id',
        'table_id',
        'room_id',
        'status',
    ];

    public function waiter()
    {
        return $this->belongsTo(Waiter::class);
    }

    public function room()
    {
        return $this->belongsTo(Room::class);
    }

    public function table()
    {
        return $this->belongsTo(Table::class);
    }

    public function customer()
    {
        return $this->belongsTo(Customer::class);
    }

    public function selling()
    {
        return $this->belongsTo(Selling::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function account()
    {
        return $this->belongsTo(Account::class);
    }

    public function soldItems()
    {
        return $this->hasMany(SoldItem::class);
    }

    public function payDebt($customerId, $amountPaid, $sellingId)
    {
        DB::beginTransaction();
        try {
            // ✅ Retrieve customer's debts oldest first
            $sales = $this->where('customer_id', $customerId)
                ->where('selling_id', $sellingId)
                ->where('due', '>', 0)
                ->orderBy('date', 'asc')
                ->get();

            foreach ($sales as $sale) {
                if ($amountPaid <= 0) {
                    break; // ✅ Stop if we’ve used all the payment
                }

                if ($amountPaid >= $sale->due) {
                    // ✅ Enough to fully pay this sale
                    $amountPaid -= $sale->due;
                    $sale->paid += $sale->due; // Add full due amount to paid
                    $sale->due = 0;
                } else {
                    // ✅ Partial payment for this sale
                    $sale->due -= $amountPaid;
                    $sale->paid += $amountPaid;
                    $amountPaid = 0;
                }

                $sale->update();

                // ✅ Double check: if fully paid, make sure status matches
                if ($sale->due == 0) {
                    $sale->paid = $sale->net; // Ensure paid == net total
                    $sale->update();
                }
            }

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to pay debt: ' . $e->getMessage());
            throw $e; // Or return a proper error response if called from a controller
        }
    }

    public function customers()
    {
        return $this->hasMany(Customer::class, 'name', 'customer');
    }
}
