<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class Purchase extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'date',
        'total',
        'discount',
        'net',
        'paid',
        'due',
        'account_id',
        'supplier_id',
        'invoice',
        'selling_id',
        'location_id',
        'status',
    ];

    public function selling()
    {
        return $this->belongsTo(Selling::class);
    }

    public function location()
    {
        return $this->belongsTo(Location::class);
    }

    public function storage()
    {
        return $this->belongsTo(Storage::class);
    }

    public function supplier()
    {
        return $this->belongsTo(Supplier::class);
    }

    public function supplierDebts()
    {
        return $this->hasMany(SupplierDebts::class);
    }

    public function account()
    {
        return $this->belongsTo(Account::class);
    }

    public function purchasedItems()
    {
        return $this->hasMany(PurchasedItem::class);
    }

    public function payDebt($supplierId, $amountPaid, $sellingId)
    {
        DB::beginTransaction();
        try {
            // ✅ Retrieve supplier's debts oldest first
            $purchases = $this->where('supplier_id', $supplierId)
                ->where('selling_id', $sellingId)
                ->where('due', '>', 0)
                ->orderBy('date', 'asc')
                ->get();

            foreach ($purchases as $purchase) {
                if ($amountPaid <= 0) {
                    break; // ✅ Stop if we’ve used all the payment
                }

                if ($amountPaid >= $purchase->due) {
                    // ✅ Enough to fully pay this purchase
                    $amountPaid -= $purchase->due;
                    $purchase->paid += $purchase->due; // Add full due amount to paid
                    $purchase->due = 0;
                } else {
                    // ✅ Partial payment for this sale
                    $purchase->due -= $amountPaid;
                    $purchase->paid += $amountPaid;
                    $amountPaid = 0;
                }

                $purchase->update();

                // ✅ Double check: if fully paid, make sure status matches
                if ($purchase->due == 0) {
                    $purchase->paid = $purchase->net; // Ensure paid == net total
                    $purchase->update();
                }
            }

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to pay debt: ' . $e->getMessage());
            throw $e; // Or return a proper error response if called from a controller
        }
    }
}
