<?php

namespace App\Http\Controllers;

use App\Models\RoleHasPermission;
use Illuminate\Http\Request;

use App\Models\User;
use Exception;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $mains = $locationDetails['mains'];
            $users = User::all();
            $roles = Role::all();
            $locationName = null;

            return view("register-users", compact(
                'users',
                'locationName',
                'sellingCount',
                'sellings',
                'mains',
                'roles'
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {;
        try {
            $request->validate([
                'name' => 'required|unique:users',
                'username' => 'required|unique:users',
                'password' => 'required|unique:users',
            ]);

            $user = new User();
            $user->name = ucwords($request->name);
            $user->role = $request->role;
            $user->phone = $request->phone;
            $user->username = $request->username;
            $user->password = Hash::make($request->password);
            $user->selling_id = $request->selling;
            $user->save();

            if ($request->role == "Admin") {
                $user->assignRole("Admin");
            } elseif ($request->role == "Manager") {
                $user->assignRole("manager");
            } elseif ($request->role == "Accountant") {
                $user->assignRole("Accountant");
            } elseif ($request->role == "Cashier") {
                $user->assignRole("Cashier");
            } elseif ($request->role == "Counter") {
                $user->assignRole("Counter");
            } elseif ($request->role == "Waiter") {
                $user->assignRole("Waiter");
            } elseif ($request->role == "Supervisor") {
                $user->assignRole("Supervisor");
            } elseif ($request->role == "Seller") {
                $user->assignRole("Seller");
            }

            return back()->with("success", "Successfully registered!!");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $selectedUser = User::findOrFail($id);

        // Remove existing roles
        DB::table("model_has_roles")->where("model_id", $id)->delete();

        // Update user fields
        $selectedUser->role = $request->role;
        $selectedUser->name = ucwords($request->name);
        $selectedUser->phone = $request->phone;
        $selectedUser->username = $request->username;
        $selectedUser->selling_id = $request->selling;

        // Only update password if a new one is provided
        if (!empty($request->password)) {
            $selectedUser->password = Hash::make($request->password);
        }

        $selectedUser->update();

        // Reassign role
        $selectedUser->assignRole($request->role);

        return back()->with("success", "Successfully updated!!");
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            User::find($id)->delete();
            return back()->with("success", "Successfully deleted!!");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function roleIndex()
    {
        try {
            $roles = Role::all();
            $permissions = Permission::all();
            return view("roles", compact("roles", "permissions"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function editRoles($id)
    {
        try {
            $roles = Role::find($id);
            $permissions = Permission::all();
            return view('edit-roles', compact("roles", "permissions", "id"))->with("success", "Successfully update");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function updateRoles(Request $request)
    {
        try {
            $role = Role::find($request->role);
            $old = RoleHasPermission::where('role_id', $request->role)->get();

            foreach ($old as $ol) {
                $permission = Permission::find($ol->permission_id);
                $role->revokePermissionTo($permission->name);
            }

            if ($request->permissions != null) {
                foreach ($request->permissions as $id) {
                    $permissions = Permission::find($id);
                    $role->givePermissionTo($permissions->name);
                }
            }

            return redirect()->back()->with("success", "Successfully update");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function newRole(Request $request)
    {
        try {
            $role = new Role();
            $role->name = $request->role;
            $role->guard_name = 'web';
            $role->save();

            return redirect()->back()->with("success", "Successfully registered");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function deleteRole($id)
    {
        try {
            $role = Role::find($id);
            $permissions = RoleHasPermission::where('role_id', $id)->get();
            foreach ($permissions as $ol) {
                $permission = Permission::find($ol->permission_id);
                $role->revokePermissionTo($permission);
            }
            $role->delete();
            return redirect()->back()->with("success", "Successfully Deleted.");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
}
