<?php

namespace App\Http\Controllers;

use App\Models\Area;
use App\Models\Plant;
use App\Models\Table;
use App\Models\TableReservation;
use App\Models\Wall;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class TableController extends Controller
{

    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellings = $locationDetails['sellings'];

            $tables = Table::all();
            $walls = Wall::all();
            $plants = Plant::all();
            $areas = Area::all();

            return view("tables", compact(
                "tables",
                "sellings",
                "walls",
                "plants",
                "areas"
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'name' => 'required|unique:tables',
                'location' => 'required',
            ]);
            foreach ($request->input('name') as $key => $value) {
                Table::create([
                    'name' => $request->input('name')[$key],
                    'selling_id' => $request->input('location')[$key],
                ]);
            }
            return back()->with("success", "Successfully Registered.");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            $table = Table::find($id);
            $table->name = $request->name;
            $table->selling_id = $request->location;
            $table->update();

            return back()->with("success", "Successfully updated.");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }


    public function updatePosition(Request $request, $type)
    {
        $request->validate([
            'id' => 'required|integer',
            'pos_x' => 'required|integer',
            'pos_y' => 'required|integer',
        ]);

        switch ($type) {
            case 'table':
                $item = Table::find($request->id);
                break;
            case 'wall':
                $item = Wall::find($request->id);
                break;
            case 'plant':
                $item = Plant::find($request->id);
                break;
            case 'area':
                $item = Area::find($request->id);
                break;
            default:
                return response()->json(['error' => 'Invalid type'], 400);
        }

        if ($item) {
            $item->pos_x = $request->pos_x;
            $item->pos_y = $request->pos_y;
            $item->save();

            return response()->json(['success' => true]);
        }

        return response()->json(['error' => 'Item not found'], 404);
    }


    public function updateWallPosition(Request $request, $type)
    {
        $model = match ($type) {
            'table' => Table::class,
            'wall' => Wall::class,
            'plant' => Plant::class,
            'area' => Area::class,
            default => abort(400, 'Invalid type'),
        };

        $record = $model::findOrFail($request->id);
        $record->update([
            'pos_x' => $request->pos_x,
            'pos_y' => $request->pos_y
        ]);

        return response()->json(['success' => true]);
    }

    public function tableReservation()
    {
        $reservations = TableReservation::with('table')
            ->orderBy('booking_start', 'asc')
            ->whereDate('booking_start', today()) // Optional: only today's bookings
            ->get();

        return view('table-reservation', compact('reservations'));
    }

    public function checkIn(TableReservation $reservation)
    {
        $reservation->update(['status' => 'checked_in']);
        return response()->json(['success' => true, 'message' => 'Customer Checked In Successfully!']);
    }

    public function cancelReservation(TableReservation $reservation)
    {
        $reservation->update(['status' => 'cancelled']);
        return response()->json(['success' => true, 'message' => 'Reservation Cancelled Successfully!']);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $table = Table::find($id);

            if (!$table) {
                return response()->json([
                    'success' => false,
                    'message' => 'Table not found.',
                ]);
            }

            $table->delete();

            return response()->json([
                'success' => true,
                'message' => 'Successfully deleted.',
            ]);
        } catch (Exception $e) {
            Log::error($e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong.',
            ], 500);
        }
    }


    public function bulkDelete(Request $request)
    {
        try {
            $ids = $request->ids;

            if (!is_array($ids) || empty($ids)) {
                return response()->json(['error' => 'No valid IDs provided.'], 400);
            }

            $deletedIds = [];

            foreach ($ids as $id) {
                $table = Table::find($id);
                if ($table) {
                    $table->delete();
                    $deletedIds[] = $id;
                }
            }

            return response()->json([
                'success' => true,
                'deleted' => $deletedIds,
                'message' => count($deletedIds) . ' table(s) deleted successfully.'
            ]);
        } catch (\Exception $e) {
            Log::error('Bulk delete error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Something went wrong during deletion.'
            ], 500);
        }
    }
}
