<?php

namespace App\Http\Controllers;

use App\Imports\SupplierImport;
use App\Models\Supplier;
use App\Models\Account;
use App\Models\BusinessProfile;
use App\Models\Selling;
use App\Models\SupplierDebts;
use App\Models\Transaction;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;

class SupplierController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $accounts = $locationDetails['accounts'];
            $suppliers = $locationDetails['suppliers'];

            return view("supplier", compact("suppliers", "sellingCount", "sellings", "accounts"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function getSupplierAdvance(Request $request)
    {
        try {
            $supplier = $request->get('supplier');
            $results = Supplier::where("name", $supplier)->first();
            return response()->json($results);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // Begin the database transaction
        DB::beginTransaction();

        try {
            $request->validate([
                'name' => 'required|unique:suppliers',
            ]);

            // Create a new Supplier instance and assign the validated data
            $supplier = new Supplier([
                'name' => ucwords($request->name),
                'address' => ucwords($request->address),
                'phone' => $request->phone,
                'email' => $request->email,
                'vrn' => $request->vrn,
                'tin' => $request->tin,
                'advance' => 0,
                'description' => '',
                'opening_balance' => $request->opening,
                'selling_id' => $request->location,
            ]);

            // Save the supplier to the database
            $supplier->save();

            if ($request->has('opening')) {
                $debt = new SupplierDebts();
                $debt->date = date('Y-m-d');
                $debt->supplier_id = $supplier->id;
                $debt->opening_balance = $request->opening;
                $debt->due = $request->opening;
                $debt->selling_id = $request->location;
                $debt->save();
            }

            // Commit the database transaction
            DB::commit();

            // Return a success response
            return back()->with("success", "Successfully added!!");
        } catch (Exception $e) {
            // An error occurred, rollback the transaction
            DB::rollBack();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        // Begin the database transaction
        DB::beginTransaction();
        try {
            $supplier = Supplier::find($id);
            $sellingId = $request->location;

            # If the selling location changes
            if ($sellingId != $supplier->selling_id) {
                SupplierDebts::where('supplier_id', $id)
                    ->where('selling_id', $supplier->selling_id)
                    ->update(['selling_id' => $sellingId]);
            }

            if ($request->opening != $supplier->opening_balance) { # Check if there is any change in oppening balance.
                $supplierDebtData = SupplierDebts::where("supplier_id", $id)
                    ->whereNotNull('opening_balance')
                    ->where("selling_id", $sellingId)
                    ->first();

                if ($supplierDebtData) {
                    # Update opening balance.
                    $supplierDebtData->opening_balance = $request->opening;
                    $supplierDebtData->update();

                    # Get the entire supplier debts.
                    $supplierDebts = SupplierDebts::where('selling_id', $sellingId)
                        ->where('supplier_id', $id)
                        ->orderBy('created_at', 'ASC')
                        ->get();

                    # Initialize variables.
                    $previousDueBalance = null;

                    # Re-calculate supplier debts.
                    foreach ($supplierDebts as $debt) {
                        // Initialize $previousDueBalance with the current debt's amount if it's null.
                        if ($previousDueBalance === null) {
                            $previousDueBalance = $debt->opening_balance + $debt->amount - $debt->paid;
                            $debt->due = $previousDueBalance;
                            $debt->selling_id = $sellingId != $debt->selling_id ? $sellingId : $debt->selling_id;
                            $debt->update();
                            continue; // Skip the rest of the loop for the first entry.
                        }

                        // Add the previous amount to the current amount
                        $debt->due = $previousDueBalance + $debt->amount - $debt->paid;
                        $debt->selling_id = $sellingId != $debt->selling_id ? $sellingId : $debt->selling_id;
                        $debt->update();

                        // Update the previous amount for the next iteration
                        $previousDueBalance = $debt->due;
                    }
                } else {
                    $debt = new SupplierDebts();
                    $debt->date = date('Y-m-d');
                    $debt->supplier_id = $supplier->id;
                    $debt->opening_balance = $request->opening;
                    $debt->due = $request->opening;
                    $debt->selling_id = $sellingId;
                    $debt->save();
                }
            }

            $supplier->name = ucwords($request->name);
            $supplier->address = ucwords($request->address);
            $supplier->phone = $request->phone;
            $supplier->email = $request->email;
            $supplier->advance = $request->advance;
            $supplier->vrn = $request->vrn;
            $supplier->tin = $request->tin;
            $supplier->opening_balance = $request->opening;
            $supplier->selling_id = $sellingId;
            $supplier->update();

            // Commit the database transaction
            DB::commit();

            return back()->with("success", "Successfully updated!!");
        } catch (Exception $e) {
            // An error occurred, rollback the transaction
            DB::rollBack();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function getSupplier(Request $request)
    {
        try {
            $query = $request->get('query');
            $response = Supplier::where('name', 'LIKE', "%$query%")
                ->take(10) // Limit the number of results to 10
                ->get();

            return response()->json($response);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function advancePayment(Request $request, $id)
    {
        try {
            $supplier = Supplier::find($id);
            $supplier->advance += $request->amount;
            $supplier->description = $request->description;
            $supplier->update();

            $account  = Account::where("name", $request->account)->first();

            $transaction = new Transaction();
            $transaction->user = auth()->user()->name;
            $transaction->date = date('Y-m-d');
            $transaction->reason = "Supplier advance";
            $transaction->status = "Out";
            $transaction->amount = $request->amount;
            $transaction->before = $account->balance;
            $transaction->after = $account->balance - $request->amount;
            $transaction->account = $request->account;
            $transaction->save();

            $account->balance -=  $request->amount;
            $account->update();

            return back()->with("success", "Successfully posted!!");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            Supplier::find($id)->delete();
            return back()->with("success", "Successfully deleted!!");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function bulkDelete(Request $request)
    {
        try {
            $ids = $request->ids;
            $results = "";
            foreach ($ids as $id) {
                $results = Supplier::find($id)->delete();
            }
            return response()->json($results);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function uploadSupplierExcel(Request $request)
    {
        // Begin the database transaction
        DB::beginTransaction();

        try {
            Excel::import(new SupplierImport, $request->file);

            // Commit the database transaction
            DB::commit();

            return back()->with("success", "Excel uploaded!!");
        } catch (Exception $e) {
            // An error occurred, rollback the transaction
            DB::rollBack();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
}
