<?php

namespace App\Http\Controllers;

use App\Models\Account;
use App\Models\Customer;
use App\Models\Debts;
use App\Models\License;
use App\Models\PostExpense;
use App\Models\Product;
use App\Models\ProductionOrder;
use App\Models\Purchase;
use App\Models\PurchaseOrder;
use App\Models\Sale;
use App\Models\Selling;
use App\Models\Service;
use App\Models\SoldItem;
use App\Models\Stock;
use App\Models\Supplier;
use App\Models\SupplierDebts;
use App\Models\Table;
use App\Models\Transaction;
use App\Models\UnitAssigned;
use App\Models\User;
use App\Models\Waiter;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Symfony\Component\Process\Process;
use Symfony\Component\Process\Exception\ProcessFailedException;

class SubscriptionController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            return view("subscription");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */

    public function store(Request $request)
    {
        try {
            // Get the MAC address of the current machine
            $mac_details = exec('getmac');
            $answer = explode(" ", $mac_details);
            $mac_address = $answer[0];

            // Retrieve the client key from the environment variables
            $client_key = env('CLIENT_KEY');

            // Get the subscription key from the request
            $key = $request->subscription;

            // Send an HTTP GET request to validate the license
            $response = Http::get('http://backroom.ajiriwa.net/validate-license/' . $key . '/' . $client_key . '/' . $mac_address);

            // $response = Http::get('http://localhost/backroom/public/validate-license/' . $key . '/' . $client_key . '/' . $mac_address);
            // return $response;


            // Check if the request was successful
            if ($response->successful()) {
                // If successful, decode the JSON response
                $result = $response->json();

                // Fetch all licenses
                $licenses = License::all();

                // Update license information based on the response
                foreach ($licenses as $license) {
                    $license->key = $result['key'];
                    $license->device = $result['device'];
                    $license->days = $result['days'];
                    $license->start_date = $result['start_date'];
                    $license->end_date = $result['end_date'];
                    $license->update();
                }

                // Extract the client ID from the response
                $client_key = $result['client_id'];

                // Return success message if license activation was successful
                return back()->with("success", "Activated successfully!!");
            } else {
                // If request was unsuccessful, return an error message
                return back()->with("failed", "Invalid activation key!!");
            }
        } catch (\Illuminate\Http\Client\RequestException $e) {
            // Log the error if an exception occurred during the request
            Log::error('Request failed: ' . $e->getMessage());
            // Return an error message if an exception occurred during the request
            return back()->with("failed", "Error occurred during request.");
        }
    }


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function seed()
    {
        Artisan::call('migrate:fresh', ['--seed' => true]);

        # Optionally, you can return a response indicating the command has been executed successfully
        return response()->json(['message' => 'Database migrated and seeded successfully.']);
    }

    public function migrate()
    {
        try {
            if (!Artisan::call('migrate:status')) {
                Artisan::call('migrate');
                return response()->json(['message' => 'Database tables migrated successfully.']);
            } else {
                return response()->json(['message' => 'No migrations to run.']);
            }
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function optimizeClear()
    {
        try {
            # Run the Artisan command
            Artisan::call('optimize:clear');

            # Retrieve the output of the command
            # $output = Artisan::output();
            # Do something with the output
            #return view('your_view', ['output' => $output]);
            return redirect()->route('login');
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function emptyTable()
    {
        try {
            # List of tables to exclude from truncation
            $excludedTables = [
                'users',
                'licenses',
                'business_profiles',
                'permissions',
                'roles'
            ];

            # Get all the table names from the database
            $tables = DB::select('SHOW TABLES');

            # Loop through each table and truncate it if it's not in the excluded list
            foreach ($tables as $table) {
                $tableName = reset((array)$table);
                if (!in_array($tableName, $excludedTables)) {
                    DB::table($tableName)->truncate();
                }
            }
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }

    public function updateLocationId()
    {
        try {
            DB::beginTransaction();

            $products = Product::all();
            $stocks = Stock::all();
            $unitAssigneds = UnitAssigned::all();

            foreach ($products as $product) {
                $product->update(['location_id' => 2]); // Update location_id to 2
                // Update the location_id in stocks and unit_assigneds tables
                $stock = $stocks->where('product_id', $product->id)->first();
                if ($stock) {
                    $stock->update(['location_id' => 2]);
                }
                $unitAssigned = $unitAssigneds->where('product_id', $product->id)->first();
                if ($unitAssigned) {
                    $unitAssigned->update(['location_id' => 2]);
                }
            }

            DB::commit();
            return response()->json(['success' => 'Location IDs updated successfully'], 200);
        } catch (Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function merge()
    {
        try {
            DB::beginTransaction();

            $sellingMap = Selling::pluck('id', 'location_id')->toArray();

            $models = [
                Account::class,
                Transaction::class,
                Supplier::class,
                Customer::class,
                PostExpense::class,
                User::class,
                Service::class,
                PurchaseOrder::class,
                Debts::class,
                SupplierDebts::class,
                Purchase::class,
                Sale::class,
                Table::class,
                Waiter::class,
                ProductionOrder::class,
            ];

            foreach ($models as $model) {
                $model::chunk(100, function ($items) use ($sellingMap) {
                    foreach ($items as $item) {
                        $sellingId = $sellingMap[$item->location_id] ?? null;
                        if ($sellingId) {
                            $item->selling_id = $sellingId;
                            $item->save();
                        }
                    }
                });
            }

            DB::commit();
            return response()->json(['success' => 'Done!'], 200);
        } catch (Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function startWebSocketServer()
    {
        $process = new Process(['php', 'artisan', 'websockets:serve']);
        $process->setTimeout(null); // Run indefinitely

        try {
            $process->start(); // Start the process asynchronously

            return response()->json([
                'success' => true,
                'message' => 'WebSocket server started in background'
            ]);
        } catch (ProcessFailedException $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    # These function cleans duplicate entries in unit assigned table
    # Specicly ment for kimry
    public function kimry()
    {
        DB::beginTransaction();

        try {
            $duplicates = DB::table('unit_assigneds')
                ->select('product_id')
                ->groupBy('product_id')
                ->havingRaw('COUNT(*) > 1')
                ->pluck('product_id');

            foreach ($duplicates as $productId) {
                // Keep the first record and delete the rest
                $ids = DB::table('unit_assigneds')
                    ->where('product_id', $productId)
                    ->orderBy('id')
                    ->pluck('id');

                $firstId = $ids->shift(); // Keep the first one

                DB::table('unit_assigneds')
                    ->where('product_id', $productId)
                    ->whereNotIn('id', [$firstId])
                    ->delete();
            }

            DB::commit();

            // For Artisan command
            //$this->info('✅ Duplicates removed. Each product_id now has only one entry.');

            // Optional: return JSON if used in a controller
            return response()->json(['status' => 'success', 'message' => 'Duplicates removed successfully']);
        } catch (\Exception $e) {
            DB::rollBack();

            // Log or print the error
            Log::error('Duplicate clean-up failed: ' . $e->getMessage());

            // For Artisan command
            //$this->error('❌ Operation failed. Check logs for details.');

            // Optional: return JSON if used in a controller
            return response()->json(['status' => 'error', 'message' => 'Something went wrong'], 500);
        }
    }

    # Clear products without names and its related stock and unit assigned details
    # This function is designed for kimry kids collection
    public function deleteUnknowProduts()
    {
        // Raw DB query to find empty/whitespace-only product names
        $products = DB::table('products')->whereRaw("LENGTH(TRIM(product)) = 0")->get();

        if ($products->isEmpty()) {
            return response()->json(['message' => 'No unnamed products found.'], 200);
        }

        DB::beginTransaction();

        try {
            foreach ($products as $product) {
                // Delete related stocks and units
                Stock::where('product_id', $product->id)->delete();
                UnitAssigned::where('product_id', $product->id)->delete();

                // Delete the actual product using Eloquent model
                Product::where('id', $product->id)->delete(); // or use Product::destroy($product->id);
            }

            DB::commit();
            return response()->json(['message' => 'Unknown products and related records cleared.'], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'Failed to clear unknown products.', 'details' => $e->getMessage()], 500);
        }
    }

    public function payment()
    {
        $licenseData = License::select(
            'location',
            'device',
            'expires_at',
            'h_expires_at'
        )->first();

        return view('payment', compact('licenseData'));
    }

    public function cleanExpiryNot()
    {
        try {
            // Update exp_not column
            $stocks = Stock::all();

            foreach ($stocks as $key => $stock) {
                # code...
                $stock->update(['exp_not' => 3]);
            }

            return response()->json(['success' => 'Expiry Dates Updated!'], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed To Update Expiry Dates: ' . $e->getMessage()], 500);
        }
    }

    public function assignDefaultBatchNumber()
    {
        try {
            $updated = Stock::whereNull('batch_no')->update(['batch_no' => '#100']);

            if ($updated) {
                return response()->json(['success' => "$updated stock(s) updated with default batch number."], 200);
            }

            return response()->json(['info' => 'No stock found without a batch number.'], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to update batch numbers.', 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Sync sales.selling_id based on product location → selling.
     * Returns stats as JSON.
     */
    public function syncSalesSellingId()
    {
        $updated = 0;
        $skipped = 0;

        DB::transaction(function () use (&$updated, &$skipped) {
            // 🔹 Step 1: Get all products with their location
            $products = Product::select('id', 'location_id')->get();

            foreach ($products as $product) {
                // 🔹 Step 2: Find selling id for that location
                $selling = Selling::where('location_id', $product->location_id)->first();

                if (!$selling) {
                    $skipped++;
                    continue;
                }

                // 🔹 Step 3: Find all sold items related to this product
                $soldItems = SoldItem::where('product_id', $product->id)->get();

                foreach ($soldItems as $soldItem) {
                    $affected = Sale::where('id', $soldItem->sale_id)
                        ->update(['selling_id' => $selling->id]);

                    if ($affected) {
                        $updated++;
                    }
                }
            }
        });

        return response()->json([
            'status'  => 'success',
            'message' => 'Sales records synced successfully.',
            'updated_sales' => $updated,
            'skipped_products' => $skipped,
        ]);
    }

    public function resetNegativeStocks()
    {
        // Get all stocks with negative qty
        $updatedCount = Stock::where('qty', '<', 0)->update(['qty' => 0]);

        return "Updated {$updatedCount} stock(s) with negative quantity to 0.";
    }

    public function getSoldItemsWithMissingSales()
    {
        // Fetch all sold_items whose sale_id is missing in sales table
        $orphanItems = SoldItem::whereNotIn('sale_id', function ($query) {
            $query->select('id')->from('sales');
        })->get();

        return $orphanItems;
    }
    public function createStorageLink()
    {
        try {
            // Run the artisan command
            Artisan::call('storage:link');

            return response()->json([
                'status' => 'success',
                'message' => 'Storage link created successfully!',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to create storage link: ' . $e->getMessage(),
            ]);
        }
    }

    public function cleaner()
    {
        $units = Stock::all();

        foreach ($units as $unit) {
            $unit->qty = 10;
            $unit->save();
        }

        return 'Done';
    }
}


# FIFO IMPLEMENTATION APPROACH
# Add batch_no column to both products, stocks, Unit_Assigned and purchases columns.
# If the product is created for the first time in any location initiate the batch_no in both products, stocks and unit-assigned tables.
# In the search query sum up the qty for each product and display it to the user.
# On sale submission loop through each sold item, get its stock data orderBy(ASC, batch_no).
# Decrement stock qty with the sold qty, if stock qty reaches 0 and there is a remainder on the sold item delete batch and deduct the remainder to the next batch.
# Do the same when transfering stock from one location to the other.
# If the sale or transfer is cancelled then reverse the process.
# On purhcase create a new product for each purchased item with a new batch_no.
# If the purchase is cancelled reverse the process.
