<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use App\Models\Room;
use App\Models\RoomPrice;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use PhpOffice\PhpSpreadsheet\Calculation\Financial\Securities\Price;

class RoomsController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            $rooms = Room::all();
            return view("rooms", compact("rooms"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function dirtyRooms()
    {
        try {
            $rooms = Room::where("reason", "cleaning")->get();
            return view("dirty-rooms", compact("rooms"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function constructionRooms()
    {
        try {
            $rooms = Room::where("reason", "construction")->get();
            return view("construction-rooms", compact("rooms"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        DB::beginTransaction();
        try {
            Room::create([
                'name' => ucwords($request->name),
                'description' => $request->description,
                'status' => 1,
                'block' => 1,
                'price' => $request->price,
            ]);
            DB::commit();
            return back()->with("success", "Room successfully added!!");
        } catch (Exception $e) {
            DB::rollBack();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            $room = Room::findOrFail($id);

            $room->update([
                'name' => ucwords($request->name),
                'description' => $request->description,
                'price' => $request->price,
            ]);

            DB::commit();

            return back()->with("success", "Successfully updated.");
        } catch (Exception $e) {
            DB::rollBack();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return back()->with("error", "Something went wrong. Please try again later.");
        }
    }

    public function updateRoomPrice(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            $room = Room::find($id);
            $roomPrices = RoomPrice::where("room", $room->name)->get();

            foreach ($roomPrices as $index => $roomPrice) {
                $roomPrice->pax = $request->pax[$index];
                $roomPrice->price = $request->price[$index];
                $roomPrice->save();
            }

            DB::commit();

            return view("rooms")->with("success", "Room price updated successfully.");
        } catch (Exception $e) {
            DB::rollBack();

            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function updateRoomPriceView($id)
    {
        try {
            $room = Room::find($id);
            $roomname = $room->name;
            $roomPrices = RoomPrice::where("room", $room->name)->get();

            return view("update-room-price", compact("roomPrices", "id", "roomname"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            DB::beginTransaction();

            $room = Room::findOrFail($id);

            if (!$room->delete()) {
                DB::rollBack();
                return response()->json(['error' => 'Something went wrong'], 500);
            }

            DB::commit();

            return back()->with("success", "Successfully deleted.");
        } catch (Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function deleteRoomPrice($id)
    {
        try {
            $roomPrice = RoomPrice::find($id);
            $room = Room::where("name", $roomPrice->room)->first();
            $roomPrice->delete();

            $roomPriceCheck = RoomPrice::where("room", $room->name)->get();
            if (count($roomPriceCheck) == 0) {
                $room->delete();
                return view("rooms")->with("success", "Room successfully deleted");
            }

            return back()->with("success", "Successfully deleted.");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function roomTransfer(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            $booking = Booking::findOrFail($id); // Get the booking details.
            $oldRoom = Room::findOrFail($booking->room_id); // Get the previous room.
            $newRoom = Room::findOrFail($request->to); // Get the new room & update its status to sold.

            $newRoom->status = "sold";
            $newRoom->save();

            // Calculate the difference in days using Carbon's diffInDays method.
            $days = Carbon::parse($booking->in_date)->diffInDays(Carbon::parse($booking->out_date));

            // Update booking details.
            $booking->room_id = $request->to;
            $booking->total = $newRoom->price * $days;
            $booking->due = $newRoom->price * $days - $booking->discount;
            $booking->save();

            // Update the previous room status to free.
            $oldRoom->status = 1;
            $oldRoom->save();

            DB::commit();

            // Redirect back with success message.
            return back()->with("success", "Successfully transfered.");
        } catch (Exception $e) {
            DB::rollBack();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function block(Request $request,  $id)
    {
        DB::beginTransaction();

        try {
            $room = Room::find($id);
            if ($request->reason == "") {
                $room->block = 1;
                $room->reason = "";
            } else {
                $room->block = 2;
                $room->reason = $request->reason;
            }

            $room->save();

            DB::commit();

            if ($request->reason == "") {
                return back()->with("success", "Successfully unblocked!!");
            } else {
                return back()->with("success", "Successfully blocked!!");
            }
        } catch (Exception $e) {
            DB::rollBack();

            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function bulkDelete(Request $request)
    {
        try {
            DB::beginTransaction();

            $ids = $request->ids;
            foreach ($ids as $id) {
                $price = Room::find($id);
                RoomPrice::where("room_id", $id)->delete();
                $price->delete();
            }

            DB::commit();
            return response()->json();
        } catch (Exception $e) {
            DB::rollBack();

            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
}
