<?php

namespace App\Http\Controllers;

use App\Models\BusinessProfile;
use App\Models\Requisition;
use App\Models\RequisitionItem;
use App\Models\Selling;
use App\Models\Supplier;
use App\Models\Unit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RequisitionController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        # Get location details
        $locationDetails = processLocationDetails();
        $sellings = $locationDetails['sellings'];

        $units = Unit::all(); # Get Units
        $suppliers = Supplier::all(); # Get Suppliers

        # Check user role
        # If admin get all requisition limit(10);
        # If get pending requisition limit(10);

        $requisitions = Requisition::with(
            'selling.location',  // eager load selling and its related location
            'items.unit',        // eager load items and their related unit
            'approvedBy'         // eager load the approvedBy relationship
        )
            ->orderBy('created_at', 'desc') // order results by latest created
            ->get();

        return view("requisitions", compact(
            "requisitions",
            "sellings",
            "units",
            "suppliers"
        ));
    }

    public function create()
    {
        # Get location details
        $locationDetails = processLocationDetails();
        $sellings = $locationDetails['sellings'];

        $units = Unit::all(); # Get Units
        $suppliers = Supplier::all(); # Get Supplier

        return view("new-requisition", compact(
            'sellings',
            'units',
            'suppliers'
        ));
    }


    public function pendingRequisition()
    {
        return view("pending-requisition");
    }
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function form()
    {
        return view("final-requisition-form");
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        DB::beginTransaction();

        try {
            # Get the selling id..
            $sellingId = Selling::where('location_id', $request->location)->first()->value('id');

            // 1️⃣ Create main requisition record
            $requisition = Requisition::create([
                'remarks' => $request->remarks,
                'selling_id' => $sellingId,
                'supplier_id' => $request->supplier,
                'requested_by' => auth()->user()->id,
                'status' => 'pending',
            ]);

            // 2️⃣ Loop and insert items using product names only
            foreach ($request->search as $index => $productName) {
                if (empty($productName)) continue; // skip empty rows

                RequisitionItem::create([
                    'requisition_id' => $requisition->id,
                    'product_name' => $productName,          // 🔹 Store the name directly
                    'unit_id' => $request->unit[$index] ?? null,
                    'qty' => $request->qty[$index] ?? 0,
                    'note' => $request->note[$index] ?? null,
                    // remove product_id completely if not needed
                ]);
            }

            DB::commit();

            return redirect()->back()->with('success', 'Requisition submitted successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Failed to submit requisition: ' . $e->getMessage());
        }
    }

    # Approving / Rejecting
    public function updateStatus(Request $request)
    {
        $request->validate([
            'status' => 'required|in:approved,rejected',
            'requisitionId' => 'required|exists:requisitions,id',
        ]);

        try {
            DB::transaction(function () use ($request) {
                Requisition::findOrFail($request->requisitionId)->update([
                    'status' => $request->status,
                    'approved_by' => auth()->id(),
                    'remarks' => $request->remarks,
                    'supplier_id' => $request->supplier,
                ]);
            });

            return redirect()->back()->with('success', 'Approved successfully');
        } catch (\Exception $e) {
            // Optionally log the error
            // Log::error('Requisition update failed', ['error' => $e->getMessage()]);

            return redirect()->back()->with('error', 'Something went wrong. Try again later.');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {

        DB::transaction(function () use ($request, $id) {
            $requisition = Requisition::findOrFail($id);

            # Get the selling id..
            $sellingId = Selling::where('location_id', $request->location)->first()->value('id');

            $requisition->update([
                'remarks' => $request->remarks,
                'selling_id' => $sellingId,
            ]);

            // Delete existing items and re-insert
            $requisition->items()->delete();

            foreach ($request->product_ids as $index => $product_id) {
                $requisition->items()->create([
                    'product_name' => $request->searchs[$index], // Optional
                    'unit_id' => $request->unit_ids[$index],
                    'qty' => $request->qtys[$index],
                    'note' => $request->notes[$index] ?? null,
                ]);
            }
        });

        return back()->with('success', 'Requisition updated successfully!');
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        $requisition = Requisition::with('items')->find($request->requisitionId);

        if (!$requisition) {
            redirect()->back()->with('error', 'Requisition Not Found..');
        }

        // Delete all items first (to maintain referential integrity)
        $requisition->items()->delete();

        // Delete the requisition itself
        $requisition->delete();

        return redirect()->back()->with('success', 'Requisition deleted successfully!');
    }

    public function bulkDelete(Request $request)
    {
        $request->validate(['ids' => 'required|array']);

        DB::transaction(function () use ($request) {
            $ids = $request->ids;

            // Soft delete items first
            RequisitionItem::whereIn('requisition_id', $ids)->delete();

            // Soft delete requisitions
            Requisition::whereIn('id', $ids)->delete();
        });

        return redirect()->back()->with('success', 'Selected requisitions archived successfully!');
    }

    public function print($requisitionId)
    {
        $requisition = Requisition::with(['requestedBy', 'approvedBy', 'items.unit'])
            ->findOrFail($requisitionId);

        $profileData = BusinessProfile::where('selling_id', $requisition->selling_id)->first();

        return view('requisition-print', compact('requisition', 'profileData'));
    }
}
