<?php

namespace App\Http\Controllers;

use App\Models\BusinessProfile;
use App\Models\PurchasedItem;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderItem;
use App\Models\Selling;
use App\Models\Supplier;
use App\Models\Unit;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PurchaseOrderController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $purchaseOrders = $locationDetails['purchaseOrders'];

            return view("purchase-order", compact("purchaseOrders"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function newPurchaseOrder($flag)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            return view("new-purchase-order", compact("sellings", "sellingCount", "flag"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getPurchaseOrder($id)
    {
        try {
            $purchaseOrderId = $id;
            $purchaseOrderData = PurchaseOrder::find($id);
            $items = PurchaseOrderItem::where("purchase_order_id", $id)->get();
            $currentDate = date('Y-m-d');
            $profileData = BusinessProfile::where("selling_id", $purchaseOrderData->selling_id)->first();
            $supplierData = Supplier::find($purchaseOrderData->supplier_id);

            return view("print-purchase-order", compact(
                "purchaseOrderId",
                "currentDate",
                "profileData",
                "items",
                "purchaseOrderData",
                "supplierData",
                "items"
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            DB::beginTransaction(); // Start a transaction
            # From the purchaseOrder invoice table get the all items with the above purchase_order_id.
            $items = PurchaseOrderItem::where("purchase_order_id", $id)->get();

            # Loop through the each item & delete it
            foreach ($items as $item) {
                $item->delete();
            }

            # Delete purchaseOrder by with the selected id.
            PurchaseOrder::find($id)->delete();

            // Commit the transaction if everything is successful
            DB::commit();

            return redirect()->back()->with("success", "Profoma deleted.");
        } catch (Exception $e) {
            // Rollback the transaction in case of an error
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function bulkDelete(Request $request)
    {
        try {
            $ids = $request->ids;
            $results = "";
            foreach ($ids as $id) {
                $results = PurchaseOrder::find($id)->delete();
            }
            return response()->json($results);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function store(Request $request, $flag, $id = null)
    {
        try {
            DB::beginTransaction();

            $sellingId = Selling::where('location_id', $request->location)->value('id');

            // Find or create supplier
            $supplier = Supplier::firstOrCreate(
                ['name' => $request->supplier],
                [
                    'address' => 'Unknown',
                    'phone' => '0717000000',
                    'opening_balance' => 0,
                    'selling_id' => $sellingId,
                ]
            );

            # ----------------------------------------------------
            #       HANDLE NEW OR EDIT MODE
            # ----------------------------------------------------

            if ($flag === 'edit' && $id !== null) {

                // Fetch existing purchase order
                $purchaseOrder = PurchaseOrder::find($id);

                if (!$purchaseOrder) {
                    return back()->with('error', 'Purchase order not found.');
                }

                // Update fields
                $purchaseOrder->date = $request->date;
                $purchaseOrder->total = $request->total;
                $purchaseOrder->supplier_id = $supplier->id;
                $purchaseOrder->selling_id = $sellingId;
                $purchaseOrder->save();

                // Delete old items
                PurchaseOrderItem::where('purchase_order_id', $id)->delete();
            } else {

                # NEW PURCHASE ORDER
                $purchaseOrder = new PurchaseOrder();
                $purchaseOrder->date = $request->date;
                $purchaseOrder->user_id = auth()->user()->id;
                $purchaseOrder->total = $request->total;
                $purchaseOrder->supplier_id = $supplier->id;
                $purchaseOrder->selling_id = $sellingId;
                $purchaseOrder->save();
            }

            # ----------------------------------------------------
            #       INSERT PURCHASE ORDER ITEMS
            # ----------------------------------------------------

            for ($i = 0; $i < count($request->id); $i++) {
                PurchaseOrderItem::create([
                    'purchase_order_id' => $purchaseOrder->id,
                    'product_id'        => $request->id[$i],
                    'unit_id'           => $request->unit[$i],
                    'cont'              => $request->cont[$i],
                    'price'             => $request->price[$i],
                    'qty'               => $request->qty[$i],
                    'amount'            => $request->amount[$i],
                ]);
            }

            # ----------------------------------------------------
            #       BUILD PRINT DATA
            # ----------------------------------------------------

            $orderArray = [
                "id" => [],
                "product" => [],
                "unit_id" => [],
                "cont" => [],
                "price" => [],
                "qty" => [],
                "amount" => [],
            ];

            function incrementProductQuantity(&$orderArray, $id, $product, $unit, $cont, $qty, $price, $amount)
            {
                $index = array_search($id, $orderArray["id"]);

                if ($index !== false) {
                    $orderArray["qty"][$index] += $qty;
                    $orderArray["amount"][$index] += $amount;
                } else {
                    $unitData = Unit::find($unit);

                    $orderArray["id"][] = $id;
                    $orderArray["product"][] = $product;
                    $orderArray["unit_id"][] = $unitData->name;
                    $orderArray["cont"][] = $cont;
                    $orderArray["price"][] = $price;
                    $orderArray["qty"][] = $qty;
                    $orderArray["amount"][] = $amount;
                }
            }

            for ($i = 0; $i < count($request->id); $i++) {
                incrementProductQuantity(
                    $orderArray,
                    $request->id[$i],
                    $request->product[$i],
                    $request->unit[$i],
                    $request->cont[$i],
                    $request->qty[$i],
                    $request->price[$i],
                    $request->amount[$i]
                );
            }

            $purchaseOrderData = PurchaseOrder::with('supplier', 'purchaseOrderItems.product', 'purchaseOrderItems.unit')
                ->find($purchaseOrder->id);

            $currentDate = date('Y-m-d');
            $profileData = BusinessProfile::where("selling_id", $sellingId)->first();
            $supplierData = $supplier;

            DB::commit();

            // Clear edit session
            session()->forget(['purchaseOrderData', 'purchaseOrderItems', 'supplierName']);

            return view("print-purchase-order", compact(
                "orderArray",
                "purchaseOrderData",
                "profileData",
                "currentDate",
                "supplierData",
            ));
        } catch (Exception $e) {

            DB::rollback();
            Log::error($e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function editPurchaseOrder($id, $flag)
    {
        $purchaseOrderData = PurchaseOrder::with('selling.location', 'supplier')->find($id);
        $purchaseOrderItems = PurchaseOrderItem::where("purchase_order_id", $id)->with('product.unit.unit', 'unit')->get();

        // Store the data in the session
        session([
            'purchaseOrderData' => $purchaseOrderData,
            'purchaseOrderItems' => $purchaseOrderItems,
            'supplierName' => $purchaseOrderData->supplier->name,
            'flag' => $flag,
        ]);

        return redirect()->route('new.purchase.order', ['flag' => $flag]);
    }

    public function clearPurchaseOrder()
    {
        $sessionKeys = ['purchaseOrderData', 'purchaseOrderItems', 'supplierName'];

        foreach ($sessionKeys as $key) {
            if (session()->has($key)) {
                session()->forget($key);
            } else {
                continue;
            }
        }

        return redirect()->route('new.purchase.order', ['flag' => 'new'])
            ->with('success', 'Cancelled Successfully');
    }
}
