<?php

namespace App\Http\Controllers;

use App\Models\BusinessProfile;
use App\Models\Customer;
use App\Models\Profoma;
use App\Models\ProfomaItems;
use App\Models\Sale;
use App\Models\Selling;
use App\Models\SoldItem;
use App\Models\Unit;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ProfomaController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            $locationDetails = processLocationDetails();
            $profomas = $locationDetails['profomas'];
            $overAllTotal = 0;

            return view("profoma-invoice-view", compact("profomas", "overAllTotal"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function newProfomaInvoice(Request $request)
    {
        try {
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            # Initiate variable
            $array = $request->all();

            if (empty($array)) {
                $status = 'new';
                return view("new-profoma-invoice", compact("sellings", "sellingCount", "status"));
            }

            # Inititate variable
            $profomaId = $array['id'];

            // Load profoma data with related trashed models
            $profomaData = Profoma::with(['customer' => function ($query) {
                $query->withTrashed();
            }])->findOrFail($profomaId);

            $profomaItems = ProfomaItems::where("profoma_id", $profomaId)
                ->with([
                    'unit' => fn($query) => $query->withTrashed(),
                    'product' => fn($query) => $query->withTrashed(),
                ])
                ->get();

            session([
                'profomaData' => $profomaData,
                'profomaItems' => $profomaItems,
                'customerName' => $profomaData->customer->name
            ]);

            $iteration = 1;
            $units = [];
            $status = 'old';

            return view("new-profoma-invoice", compact(
                "sellings",
                "sellingCount",
                "iteration",
                "units",
                "status",
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, $status)
    {
        try {
            DB::beginTransaction(); # Start a transaction

            # Get selling data
            $sellingId = Selling::where("location_id", $request->location)->first()->value('id');

            # Initiate varibale.
            $customerId = null;
            $profomaId = null;
            $profoma = collect();

            # Get or create the customer
            $customer = Customer::firstOrCreate(
                ['name' => $request->customer],
                [
                    'phone' => $request->phone,
                    'address' => $request->address,
                    'limit' => 0,
                    'opening_balance' => 0,
                    'selling_id' => $sellingId,
                ]
            );

            $customerId = $customer->id;

            # Shared profoma fields
            $profomaFields = [
                'user_id'     => auth()->id(),
                'date'        => $request->date,
                'selling_id'  => $sellingId,
                'customer_id' => $customerId,
                'total'       => $request->total,
            ];

            if ($status == "old") {
                # Retrieves session
                $profomaSession = session('profomaData');

                # Update existing profoma
                $profoma = Profoma::with("customer")->findOrFail($profomaSession->id);
                $profoma->update($profomaFields);
                $profomaId = $profomaSession->id;

                # Delete old items
                ProfomaItems::where("profoma_id", $profomaId)->delete();
            } else {
                # Create new profoma
                $profoma = Profoma::create($profomaFields);
                $profomaId = $profoma->id;
            }

            # Loop through the items.
            for ($i = 0; $i < count($request->id); ++$i) {
                $itemsData[] = [
                    'profoma_id' => $profomaId,
                    'product_id' => $request->id[$i],
                    'unit_id'    => $request->unit[$i],
                    'cont'       => $request->cont[$i],
                    'price'      => $request->price[$i],
                    'qty'        => $request->qty[$i],
                    'amount'     => $request->amount[$i],
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }

            # Insert profoma items
            $items = ProfomaItems::insert($itemsData);

            $profileData = BusinessProfile::where('selling_id', $sellingId)->first();

            // Commit the transaction if everything is successful
            DB::commit();

            session()->forget(['profomaData', 'profomaItems', 'customerName']);

            return view("print-profoma-invoice", compact(
                "profoma",
                "profileData",
                "items"
            ));
        } catch (Exception $e) {
            // Rollback the transaction in case of an error
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            DB::beginTransaction(); // Start a transaction
            # From the profoma invoice table get the all items with the above profoma_id.
            $items = ProfomaItems::where("profoma_id", $id)->get();

            # Loop through the each item & delete it
            foreach ($items as $item) {
                $item->delete();
            }

            # Delete profoma by with the selected id.
            Profoma::find($id)->delete();

            // Commit the transaction if everything is successful
            DB::commit();

            return redirect()->route('profoma.invoice.view')->with("success", "Profoma deleted.");
        } catch (Exception $e) {
            // Rollback the transaction in case of an error
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function getProfoma($id)
    {
        try {
            $profomaId = $id;
            $profoma = Profoma::find($profomaId);
            $items = ProfomaItems::where("profoma_id", $id)->get();
            $currentdate = date('Y-m-d');
            $profileData = BusinessProfile::where('selling_id', $profoma->selling_id)->first();
            $customerData = Customer::find($profoma->customer_id);

            return view("print-profoma-invoice", compact(
                "profomaId",
                "currentdate",
                "profileData",
                "items",
                "profoma",
                "customerData"
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function printDelivery($id)
    {
        try {
            $sale = Sale::where("id", $id)->with("customer")->first();
            $items = SoldItem::where("sale_id", $id)->with("product", "unit")->get();
            $sellingData = Selling::where("location_id", $sale->location_id)->first();
            $profileData = BusinessProfile::select("name", "address", "phone", "email")
                ->where("main_location_id", $sellingData->main_location_id)
                ->first();

            return view('print-delivery', compact("id", "sale", "items", "profileData"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function cancelProfomaInvoice()
    {
        $sessionKeys = ['profomaData', 'profomaItems', 'customerName'];

        foreach ($sessionKeys as $key) {
            if (session()->has($key)) {
                session()->forget($key);
            } else {
                continue;
            }
        }

        return redirect()->route('new.profoma.invoice')->with('success', 'Cancelled successfully.');
    }

    public function bulkDelete(Request $request)
    {
        try {
            $ids = $request->ids;
            $results = "";
            foreach ($ids as $id) {
                $results = Profoma::find($id)->delete();
            }
            return response()->json($results);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
}
