<?php

namespace App\Http\Controllers;

use App\Imports\ExpenseImport;
use Illuminate\Http\Request;

use App\Models\PostExpense;
use App\Models\Account;
use App\Models\Transaction;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;

class PostExpenseController extends Controller
{

    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $accounts = $locationDetails['accounts'];

            return view('post-expense', compact("sellings", "sellingCount", "accounts"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function uploadExpenseExcel(Request $request)
    {
        try {
            Excel::import(new ExpenseImport, $request->file);

            return back()->with("success", "Excel uploaded!!");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // Begin a database transaction to ensure consistency
        DB::beginTransaction();

        try {
            $accountData = []; // Keep track of processed accounts

            for ($i = 0; $i < count($request->cat); ++$i) {
                // Save the posted expense
                $expense = new PostExpense();
                $expense->date = $request->date;
                $expense->user_id = auth()->user()->id;
                $expense->selling_id = $request->location;
                $expense->category_id = $request->cat[$i];
                $expense->description = $request->desc[$i];
                $expense->sum = $request->sum[$i];
                $expense->total = $request->total;
                $expense->account_id = $request->account[$i];
                $expense->save();

                // Handle account update
                $accountId = $request->account[$i];
                $amount = $request->sum[$i];
                $account = Account::find($accountId);

                if (!$account) {
                    throw new Exception("Account not found with ID: {$accountId}");
                }

                $beforeBalance = $account->balance;

                // Deduct expense from account balance
                $account->balance -= $amount;
                $account->save();

                // Record the transaction using the reusable helper
                recordTransaction(
                    $request->date,      // date
                    "Expense",           // reason
                    "Out",               // status
                    $amount,             // amount
                    $account->id,        // accountId
                    auth()->id()         // userId
                );

                // Optionally track processed accounts (if needed later)
                $accountData[$accountId] = $account->balance;
            }

            // Commit the database transaction
            DB::commit();

            return back()->with("success", "Expenses posted successfully.");
        } catch (\Exception $e) {
            // Rollback in case of any error
            DB::rollBack();
            Log::error($e->getMessage());

            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function bulkDelete(Request $request)
    {
        DB::beginTransaction();
        try {
            $ids = $request->ids;

            foreach ($ids as $id) {
                $expense = PostExpense::find($id);
                if (!$expense) continue;

                $account = Account::find($expense->account_id);
                if (!$account) continue;

                $amount = $expense->sum;

                // Use reusable helper to record transaction
                recordTransaction(
                    date('Y-m-d'),
                    'Expense deleted',
                    'In',
                    $amount,
                    $account->id,
                    auth()->user()->id,
                );

                // Update account balance
                $account->balance += $amount;
                $account->save();

                // Delete the expense
                $expense->delete();
            }

            DB::commit();
            return response()->json(['success' => 'Expenses reversed successfully']);
        } catch (Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function changeDate(Request $request)
    {
        // Start database transaction
        DB::beginTransaction();

        try {
            $ids = $request->get("ids");         // Array of PostExpense IDs
            $newDate = $request->get("newdate"); // New date to set

            // Retrieve all selected expenses
            $expenses = PostExpense::whereIn('id', $ids)->get();

            foreach ($expenses as $expense) {
                // Update the expense date
                $expense->update(['date' => $newDate]);

                // Update the corresponding transaction(s) for this expense
                Transaction::where('account_id', $expense->account_id)
                    ->where('selling_id', $expense->selling_id)
                    ->where('reason', 'Expense')
                    ->where('amount', $expense->sum)
                    ->whereDate('date', $expense->date) // Match previous expense date
                    ->update(['date' => $newDate]);
            }

            DB::commit();
            return response()->json(['success' => 'Date updated successfully']);
        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error updating expense dates: ' . $e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
}
