<?php

namespace App\Http\Controllers;

use App\Imports\MenuImport;
use App\Models\BusinessProfile;
use App\Models\Menu;
use App\Models\MenuItems;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\Selling;
use App\Models\Stock;
use App\Models\StockMovement;
use App\Models\Unit;
use App\Models\UnitAssigned;
use App\Models\Waiter;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Storage; // <-- Add this line

class MenuController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function menuIndex()
    {
        // Get location details
        $locationDetails = processLocationDetails();
        $sellingCount = $locationDetails['sellingCount'];
        $sellings = $locationDetails['sellings'];

        // Get all category IDs under parent_id = 2
        $categories = ProductCategory::where('id', 1)
            ->orWhere('parent_id', 1)->get();

        $categoryIds = $categories->pluck('id')->toArray();

        // Get products in those categories with their units
        $menus = Product::whereIn('category_id', $categoryIds)
            ->with('unit.unit')
            ->get();

        // Get all units
        $units = Unit::all();

        // Return the view
        return view('food', compact('menus', 'sellings', 'units', 'categories'));
    }

    public function drinkIndex()
    {
        # Get location details
        $locationDetails = processLocationDetails();
        $sellingCount = $locationDetails['sellingCount'];
        $sellings = $locationDetails['sellings'];

        // Get all category IDs under parent_id = 2
        $categories = ProductCategory::where('id', 2)
            ->orWhere('parent_id', 2)->get();

        $categoryIds = $categories->pluck('id')->toArray();

        $menus = Product::whereIn('location_id', $sellings->pluck('location_id')->toArray())->whereIn('category_id', $categoryIds)
            ->with('stock', 'unit.unit')
            ->get();

        # Get all units
        $units = Unit::all();

        # Return the view with the menus, locations and units
        return view("drinks", compact("menus", "sellings", "units", "categories"));
    }

    public function assignIngridientsView($id)
    {
        $product = Product::find($id);
        $units = Unit::all();
        $menuItems = MenuItems::where("product_id", $id)
            ->with("unit", "menu", "ingredient")->get();

        return view("menu-ingrideints", compact("product", "units", "menuItems"));
    }

    public function uploadMenus(Request $request)
    {
        Excel::import(new MenuImport, $request->file);
        return back()->with("success", "Excel successfully uploaded.");
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */

    public function storeMenu(Request $request)
    {
        DB::beginTransaction();

        try {
            // Validate the request
            $validated = $request->validate([
                'menu' => 'required',
                'selling' => 'required',
                'unit' => 'required',
                'location' => 'required',
            ]);

            // Get locationId
            $locationId = $request->location;

            // Check if menu already exists in this location
            $existingProduct = Product::where('product', ucwords($request->menu))
                ->where('location_id', $locationId)
                ->first();

            if ($existingProduct) {
                return back()->with('error', 'Menu already exists.');
            }

            // Initialize the image file name
            $fileName = '';

            # Image codes
            if (request()->image) {
                if ($request->hasFile('image')) {
                    $image = $request->file('image');
                    $fileName = uniqid('image_') . '.' . $image->getClientOriginalExtension();
                    // Store the image in the public disk
                    $image->storeAs('public/images', $fileName);
                }
            }

            // Insert product
            $product = new Product();
            $product->image = $fileName ?? ''; // If there is a file name the
            $product->product = ucwords($request->menu);
            $product->category_id = $request->category ?? 1; // Default to category ID 1 (Menu) if not provided
            $product->buying = 0;
            $product->location_id = $locationId;
            $product->save();

            // instret stock
            $stock = new Stock();
            $stock->product_id = $product->id;
            $stock->location_id = $locationId;
            $stock->unit_id = $request->unit;
            $stock->qty = 0;
            $stock->min_qty = 0;
            $stock->save();

            // Insert unit assignment
            $unit = new UnitAssigned();
            $unit->product_id = $product->id;
            $unit->unit_id = $request->unit;
            $unit->unit_cont = 1;
            $unit->selling = $request->selling;
            $unit->location_id = $locationId;
            $unit->save();

            DB::commit();

            return back()->with('success', 'Successfully registered.');
        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Menu store failed: ' . $e->getMessage());

            return back()->with('error', 'An error occurred while processing the request.');
        }
    }

    public function storeDrink(Request $request)
    {
        DB::beginTransaction();

        try {
            $validated = $request->validate([
                'name' => 'required|string',
                'buying' => 'required|numeric',
                'selling' => 'required|numeric',
                'unit' => 'required|integer',
                'qty' => 'required|numeric',
                'min-qty' => 'required|numeric',
                'location' => 'required|integer',
            ]);;

            // Insert product
            $product = new Product();
            $product->product = ucwords($request->name);
            $product->category_id = $request->category ?? 2; // Default to category ID 2 (Drink) if not provided
            $product->buying = $request->buying;
            $product->location_id = $request->location;
            $product->save();

            // Insert stock
            recordStockMovement(
                date('Y-m-d'),
                Auth::id(),
                $product->id,
                $request->location,
                "Registered",
                "In",
                $request->qty
            );

            // Insert unit assignment
            $unit = new UnitAssigned();
            $unit->product_id = $product->id;
            $unit->unit_id = $request->unit;
            $unit->unit_cont = 1; // Optional: make dynamic if needed
            $unit->selling = $request->selling;
            $unit->location_id = $request->location;
            $unit->save();

            DB::commit();
            return back()->with('success', 'Successfully Registered.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Drink store failed: ' . $e->getMessage());
            return back()->with('error', 'An error occurred while processing the request.');
        }
    }

    public function store(Request $request)
    {
        DB::beginTransaction(); // Start a transaction
        try {
            # Abort if the product already exists.
            if (Product::where('product', ucwords($request->product))->where('location_id', $request->location)->exists()) {
                return back()->with('error', "{$request->product} already exists in {$request->location}");
            }

            # Get the selected category from categories table.
            $category = ProductCategory::find($request->category);
            # If the product is a menu
            if ($category->name == "Menu") {
                // Initialize the image file name
                $request->validate([
                    'category' => 'required',
                    'product' => 'required',
                    'unit' => 'required',
                    'content' => 'required',
                    'selling' => 'required',
                    'location' => 'required'
                ]);

                # Insert product details
                $product = new Product();
                $product->barcode = "";
                $product->product = ucwords($request->product);
                $product->category = $request->category;
                $product->buying = "";
                $product->supplier_id = "";
                $product->location_id = $request->location;
                $product->save();

                for ($i = 0; $i < count($request->unit); ++$i) {
                    # Insert unit assigned details
                    $unit = new UnitAssigned();
                    $unit->product_id = $product->id;
                    $unit->product = ucwords($request->product);
                    $unit->unit_id = $request->unit[$i];
                    $unit->unit_cont = $request->content[$i];
                    $unit->selling = $request->selling[$i];
                    $unit->location_id = $request->location;
                    $unit->save();
                }
            } else {
                // Initialize the image file name
                $fileName = '';

                # Image codes
                if (request()->image) {
                    if ($request->hasFile('image')) {
                        $image = $request->file('image');
                        $fileName = uniqid('image_') . '.' . $image->getClientOriginalExtension();
                        // Store the image in the public disk
                        $image->storeAs('public/images', $fileName);
                    }
                }

                # If the product is a normal product...insert product details
                $product = new Product();
                $product->barcode = $request->barcode != null ? $request->barcode : str_pad(rand(0, 9999999999999), 13, '0', STR_PAD_LEFT);
                $product->product = ucwords($request->product);
                $product->image = $fileName ?? ''; // If there is a file name the insert it if not insert an empty string.
                $product->category_id = $request->category;
                $product->buying = $request->buying;
                $product->supplier_id = $request->supplier;
                $product->location_id = $request->location;
                $product->save();

                for ($i = 0; $i < count($request->unit); ++$i) {
                    # Insert unit_assigned details
                    $unit = new UnitAssigned();
                    $unit->product_id = $product->id;
                    $unit->unit_id = $request->unit[$i];
                    $unit->unit_cont = $request->content[$i];
                    $unit->selling = $request->selling[$i];
                    $unit->location_id = $request->location;
                    $unit->save();
                }
                # Insert stock details
                $stock = new Stock();
                $stock->product_id = $product->id;
                $stock->location_id = $request->location;
                $stock->unit_id = $request->minunit;
                $stock->min_cont = $request->mincont;
                $stock->min_qty = $request->minqty;
                $stock->qty = $request->qty;
                $stock->exp = $request->exp;
                $stock->exp_not = $request->expnot;
                $stock->save();
            }

            // Commit the transaction if everything is successful
            DB::commit();

            return back()->with("success", "Product successfully Registered!!");
        } catch (Exception $e) {
            // Rollback the transaction in case of an error
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */

    public function assignIngridients(Request $request, $id)
    {
        // Find the menu or fail if not found
        $menu = Menu::findOrFail($id);

        // Retrieve input arrays
        $ingredientIds = $request->input('id');
        $unitIds = $request->input('unit-id');
        $qtys = $request->input('qty');

        // Check for already assigned ingredients
        $existingIngredients = MenuItems::where('menu_id', $menu->id)
            ->whereIn('ingredient_id', $ingredientIds)
            ->pluck('ingredient_id')
            ->toArray();

        $newItems = [];

        DB::beginTransaction();

        try {
            for ($i = 0; $i < count($ingredientIds); ++$i) {
                $ingredientId = $ingredientIds[$i];
                $qty = $qtys[$i];

                if (!in_array($ingredientId, $existingIngredients)) {
                    $newItems[] = [
                        'menu_id' => $menu->id,
                        'ingredient_id' => $ingredientId,
                        'qty' => $qty,
                        'created_at' => now(),
                        'updated_at' => now()
                    ];
                }
            }

            if (!empty($newItems)) {
                MenuItems::insert($newItems);
            }

            DB::commit();

            return back()->with('success', 'Assigned successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Assign Ingredients Error: " . $e->getMessage());
            return back()->with('error', 'An error occurred.');
        }
    }



    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */

    public function updateMenu(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            // Validate the request
            $validated = $request->validate([
                'menu' => 'required',
                'selling' => 'required',
                'unit' => 'required',
                'location' => 'required',
            ]);

            // Find the existing product
            $product = Product::findOrFail($id);

            $locationId = $request->location;

            // Handle image update if a new image is uploaded
            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $fileName = uniqid('image_') . '.' . $image->getClientOriginalExtension();
                $image->storeAs('public/images', $fileName);

                // Delete old image if exists
                if ($product->image && Storage::exists('public/images/' . $product->image)) {
                    Storage::delete('public/images/' . $product->image);
                }

                $product->image = $fileName;
            }

            // Update product fields
            $product->product = ucwords($request->menu);
            $product->category_id = $request->category;
            $product->location_id = $locationId;
            $product->save();

            // Update stock (if needed, e.g., unit change)
            $stock = Stock::where('product_id', $product->id)
                ->where('location_id', $locationId)
                ->first();

            if ($stock) {
                $stock->unit_id = $request->unit;
                $stock->save();
            }

            // Update unit assignment
            $unitAssigned = UnitAssigned::where('product_id', $product->id)
                ->where('location_id', $locationId)
                ->first();

            if ($unitAssigned) {
                $unitAssigned->unit_id = $request->unit;
                $unitAssigned->selling = $request->selling;
                $unitAssigned->save();
            }

            DB::commit();

            return back()->with('success', 'Menu updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Menu update failed: ' . $e->getMessage());

            return back()->with('error', 'An error occurred while updating the menu.');
        }
    }

    public function updateDrink(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            $validated = $request->validate([
                'name' => 'required|string',
                'buying' => 'required|numeric',
                'selling' => 'required|numeric',
                'unit' => 'required|integer',
                'qty' => 'required|numeric',
                'min-qty' => 'required|numeric',
                'location' => 'required|integer',
            ]);

            // Fetch the existing product
            $product = Product::findOrFail($id);

            // Update product details
            $product->product = ucwords($request->name);
            $product->buying = $request->buying;
            $product->location_id = $request->location;
            $product->save();

            // Update or create stock record
            $stock = Stock::where('product_id', $product->id)
                ->where('location_id', $request->location)
                ->first();

            $beforeQty = $stock ? $stock->qty : 0;

            if ($stock) {
                $stock->unit_id = $request->unit;
                $stock->qty = $request->qty;
                $stock->min_qty = $request->input('min-qty');
                $stock->save();
            } else {
                $stock = new Stock([
                    'product_id' => $product->id,
                    'location_id' => $request->location,
                    'unit_id' => $request->unit,
                    'qty' => $request->qty,
                    'min_qty' => $request->input('min-qty')
                ]);
                $stock->save();
            }

            // Update or create unit assignment
            $unit = UnitAssigned::where('product_id', $product->id)
                ->where('unit_id', $request->unit)
                ->where('location_id', $request->location)
                ->first();

            if ($unit) {
                $unit->selling = $request->selling;
                $unit->unit_cont = 1; // Optional: adjust if needed
                $unit->save();
            } else {
                UnitAssigned::create([
                    'product_id' => $product->id,
                    'unit_id' => $request->unit,
                    'unit_cont' => 1,
                    'selling' => $request->selling,
                    'location_id' => $request->location
                ]);
            }

            // Log stock movement only if quantity changed
            if ($beforeQty != $request->qty) {
                StockMovement::create([
                    'date' => now()->format('Y-m-d'),
                    'user_id' => auth()->id(),
                    'product_id' => $product->id,
                    'reason' => 'Updated',
                    'status' => $request->qty > $beforeQty ? 'in' : 'out',
                    'qty' => abs($request->qty - $beforeQty),
                    'before' => $beforeQty,
                    'after' => $request->qty,
                    'location_id' => $request->location
                ]);
            }

            DB::commit();
            return back()->with('success', 'Product updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Drink update failed: ' . $e->getMessage());
            return back()->with('error', 'Failed to update the product.');
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */

    public function destroyMenu($id)
    {
        DB::beginTransaction();

        try {
            // Find the product
            $product = Product::findOrFail($id);

            // Delete related stock records
            Stock::where('product_id', $product->id)->delete();

            // Delete related unit assignments
            UnitAssigned::where('product_id', $product->id)->delete();

            // Delete product image if exists
            if ($product->image && Storage::exists('public/images/' . $product->image)) {
                Storage::delete('public/images/' . $product->image);
            }

            // Delete the product itself
            $product->delete();

            DB::commit();

            return back()->with('success', 'Menu deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Menu delete failed: ' . $e->getMessage());

            return back()->with('error', 'Failed to delete the menu. Please try again.');
        }
    }

    public function destroyDrink($id)
    {
        try {
            DB::beginTransaction();

            $menu = Menu::findOrFail($id);

            // Delete related menu items
            MenuItems::where('menu_id', $menu->id)->delete();

            // Delete the image from storage if it exists
            if ($menu->image && file_exists(public_path('storage/images/' . $menu->image))) {
                unlink(public_path('storage/images/' . $menu->image));
            }

            // Delete the menu itself
            $menu->delete();

            DB::commit();

            return back()->with('success', 'Successfully deleted.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Menu deletion failed: ' . $e->getMessage());
            return back()->with('error', 'An error occurred.');
        }
    }

    public function bulkDelete(Request $request)
    {
        $ids = $request->ids;

        try {
            DB::beginTransaction();

            foreach ($ids as $id) {
                $menu = Menu::findOrFail($id);

                // Delete related menu items
                MenuItems::where('menu_id', $menu->id)->delete();

                // Delete the image from storage if it exists
                if ($menu->image && file_exists(public_path('storage/images/' . $menu->image))) {
                    unlink(public_path('storage/images/' . $menu->image));
                }

                // Delete the menu itself
                $menu->delete();
            }

            DB::commit();

            return back()->with('success', 'Successfully deleted.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Menu deletion failed: ' . $e->getMessage());
            return back()->with('error', 'An error occurred.');
        }
    }

    public function getMenuByCategory(Request $request)
    {
        try {
            // Validate request inputs
            $request->validate([
                'categoryId' => 'required|integer',
                'waiterId' => 'required|integer',
            ]);

            $categoryId = $request->get('categoryId');
            $waiterId = $request->get('waiterId');

            $waiter = Waiter::find($waiterId);
            if (!$waiter) {
                return response()->json(['error' => 'Waiter not found'], 404);
            }

            $sellingId = $waiter->selling_id;
            $locationId = Selling::find($sellingId)->value("location_id");

            $businessProfile = BusinessProfile::where("selling_id", $sellingId)->first();
            $businessName = $businessProfile ? $businessProfile->name : null;

            $menus = Product::where('category_id', $categoryId)
                ->where('location_id', $locationId)
                ->with('unit.unit', 'category', 'location')
                ->take(20)
                ->get();

            return response()->json([
                'menus' => $menus,
                'business_name' => $businessName,
            ]);
        } catch (Exception $e) {
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function getCategoryMenu(Request $request)
    {
        try {
            // Validate request inputs
            $request->validate([
                'query' => 'required',
                'categoryId' => 'required|integer',
                'waiterId' => 'required|integer',
            ]);

            $query = $request->get('query');
            $categoryId = $request->get('categoryId');
            $waiterId = $request->get('waiterId');

            $waiter = Waiter::find($waiterId);
            if (!$waiter) {
                return response()->json(['error' => 'Waiter not found'], 404);
            }

            $sellingId = $waiter->selling_id;
            $locationId = Selling::find($sellingId)->value("location_id");

            $businessProfile = BusinessProfile::where("selling_id", $sellingId)->first();
            $businessName = $businessProfile ? $businessProfile->name : null;

            $menus = Product::where('category_id', $categoryId)
                ->where('location_id', $locationId)
                ->where('product', 'like', '%' . $query . '%')
                ->with(['unit.unit', 'category', 'location'])
                ->take(20)
                ->get();

            return response()->json([
                'menus' => $menus,
                'business_name' => $businessName,
            ]);

            // Return the results and business name as a JSON response
            return response()->json([$menus, $businessName]);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
}
