<?php

namespace App\Http\Controllers;

use App\Models\Account;
use App\Models\BusinessProfile;
use App\Models\License;
use App\Models\LoadVehicle;
use App\Models\Location;
use App\Models\Main;
use App\Models\Product;
use App\Models\Selling;
use App\Models\Stock;
use App\Models\Storage;
use App\Models\UnitAssigned;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class LocationController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            $locationDetails = processLocationDetails();
            $mainLocations = $locationDetails['mains'];
            $sellingLocations = $locationDetails['sellings'];
            $all_locations = Location::all();

            return view("location", compact(
                'mainLocations',
                'sellingLocations',
                'all_locations'
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */

    public function store(Request $request)
    {
        try {
            DB::transaction(function () use ($request) {
                // Validate input
                $request->validate([
                    'name' => 'required|string|unique:locations,name',
                    'type' => 'required|in:main,selling,storage',
                    'owner' => 'nullable|string',
                    'phone' => 'nullable|string',
                    'status' => 'required|in:static,dynamic',
                    'main' => 'nullable|exists:mains,id',
                    'selling' => 'nullable|exists:sellings,id',
                ]);

                // Check license limit if type is 'selling'
                if ($request->type === 'selling') {
                    $licenseData = License::find(1);
                    $sellingLocationCount = Selling::count();

                    if ($licenseData && $sellingLocationCount >= (int) $licenseData->location) {
                        throw new \Exception("Your license permits only " . $licenseData->location . " selling location(s).");
                    }
                }

                // Create base location
                $location = new Location();
                $location->name = ucwords($request->name);
                $location->owner = $request->owner ? ucwords($request->owner) : null;
                $location->phone = $request->phone;
                $location->type = $request->type;
                $location->status = $request->status;
                $location->save();

                // Handle type-specific records
                if ($request->type === "main") {
                    $main = new Main();
                    $main->location_id = $location->id;
                    $main->save();
                } elseif ($request->type === "selling") {
                    $selling = new Selling();
                    $selling->location_id = $location->id;
                    $selling->main_id = $request->main;
                    $selling->save();

                    $profile = new BusinessProfile();
                    $profile->selling_id = $selling->id;
                    $profile->address = 'Unknown';
                    $profile->save();

                    $account = new Account();
                    $account->name = "Cash Account";
                    $account->type = "Cash Account";
                    $account->number = "0000";
                    $account->balance = 0;
                    $account->selling_id = $selling->id;
                    $account->save();
                } elseif ($request->type === "storage") {
                    $storage = new Storage();
                    $storage->location_id = $location->id;
                    $storage->selling_id = $request->selling;
                    $storage->save();
                }
            });

            return back()->with("success", "Location added successfully.");
        } catch (\Exception $e) {
            Log::error('Location Store Error: ' . $e->getMessage(), ['exception' => $e]);
            return back()->with("error", $e->getMessage() ?: "Something went wrong. Please try again.");
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */

    public function update(Request $request, $id)
    {
        try {
            return DB::transaction(function () use ($request, $id) {
                // Find the location
                $location = Location::find($id);

                // Update the common location details
                $location->name = ucwords($request->name);
                $location->owner = ucwords($request->owner);
                $location->phone = $request->phone;
                $location->type = $request->type;
                $location->status = $request->status;
                $location->update();

                $mian = null;

                // Handle specific cases based on the request type
                if ($request->type === "main") {
                    $main = new main();
                    $main->location_id = $id;
                    $main->save();
                } elseif ($request->type === "selling") {
                    // Find the associated selling record and update its selling location
                    $sell = Selling::where('location_id', $id)->first();
                    if ($sell) {
                        $sell->selling_location = ucwords($request->name);
                        $sell->update();
                    } else {
                        $profile = BusinessProfile::where("name", ucwords($request->main))->first();
                        $newSellingLocation = new Selling();
                        $newSellingLocation->location_id = $id;
                        $newSellingLocation->main_location_id = $profile->id;
                        $newSellingLocation->selling_location = ucwords($request->name);
                        $newSellingLocation->save();

                        $oldStorageLocation = Storage::where("location_id", $id)->first();
                        if ($oldStorageLocation) {
                            $oldStorageLocation->delete();
                        }
                    }
                } elseif ($request->type === "storage") {
                    // Find the associated storage record and update its storage location
                    $store = Storage::where('location_id', $id)->first();
                    if ($store) {
                        $store->storage_location = ucwords($request->name);
                        $store->update();
                    } else {
                        $sellingData = Selling::where("location_id", $request->selling)->first();
                        $newStorageLocation = new Storage();
                        $newStorageLocation->location_id = $id;
                        $newStorageLocation->selling_id = $sellingData == null ? 1 : $sellingData->id;
                        $newStorageLocation->storage_location = ucwords($request->name);
                        $newStorageLocation->save();

                        $oldSellingLocation = Selling::where("location_id", $id)->first();
                        if ($oldSellingLocation) {
                            $oldSellingLocation->delete();
                        }
                    }
                }

                // Return a success message
                return back()->with('success', 'Successfully updated.');
            });
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */

    public function destroy($id)
    {
        try {
            return DB::transaction(function () use ($id) {
                // Find the deleted location
                $location = Location::find($id);

                // Delete associated products
                Product::where("location_id", $location->id)->delete();

                // Delete associated stocks
                Stock::where("location_id", $location->id)->delete();

                // Delete associated units_assigned
                UnitAssigned::where("location_id", $location->id)->delete();

                if ($location->type === "selling") {
                    Selling::where("location_id", $location->id)->delete();
                    Account::where("selling_location_id", $location->id)->delete();
                } elseif ($location->type === "storage") {
                    Storage::where("location_id", $location->id)->delete();
                } elseif ($location->type === "main") {
                    // Delete associated business profiles
                    BusinessProfile::where("main_location_id", $location->id)->delete();

                    // Delete associated selling records
                    Selling::where("main_location_id", $location->id)->delete();

                    // Delete associated storage records
                    Storage::where("location_id", $location->id)->delete();
                }

                // Finally, delete the location
                $location->delete();

                // Return a success message
                return back()->with("success", "Successfully deleted.");
            });
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function bulkDelete(Request $request)
    {
        try {
            return DB::transaction(function () use ($request) {
                $ids = $request->ids;

                foreach ($ids as $id) {
                    $location = Location::find($id);

                    // Delete associated products
                    Product::where("location", $location->name)->delete();

                    // Delete associated stocks
                    Stock::where("location", $location->name)->delete();

                    // Delete associated units_assigned
                    UnitAssigned::where("location", $location->name)->delete();

                    if ($location->type === "selling") {
                        Selling::where("location_id", $location->id)->delete();
                        Account::where("selling_location_id", $location->id)->delete();
                    } elseif ($location->type === "storage") {
                        Storage::where("location_id", $location->id)->delete();
                    } elseif ($location->type === "main") {
                        // Delete associated business profiles
                        BusinessProfile::where("main_location_id", $location->id)->delete();

                        // Delete associated selling records
                        Selling::where("main_location_id", $location->id)->delete();

                        // Delete associated storage records
                        Storage::where("location_id", $location->id)->delete();
                    }

                    // Delete the location
                    $location->delete();
                }

                return response()->json([]);
            });
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function locationStatus(Request $request)
    {
        $locationId = $request->get('location');

        // Get sellingId directly without extra ->first()->value()
        $sellingId = Selling::where('location_id', $locationId)->value('id');

        // Get status directly without loading the full Location model
        $status = Location::where('id', $locationId)->value('status');

        // Get salesman in a single query or null if not exists
        $salesman = LoadVehicle::where('selling_id', $sellingId)->value('salesmen_id') ?? '';

        return response()->json([$status, $salesman]);
    }
}
