<?php

namespace App\Http\Controllers;

use App\Models\BusinessProfile;
use App\Models\Customer;
use App\Models\Damage;
use App\Models\Debts;
use App\Models\Location;
use App\Models\PostExpense;
use App\Models\Product;
use Carbon\Carbon;
use App\Models\Purchase;
use App\Models\Requisition;
use App\Models\Sale;
use App\Models\Selling;
use App\Models\Stock;
use App\Models\SupplierDebts;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class HomeController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */

    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */

    public function index()
    {
        if (auth()->user()->role === 'Waiter') {
            return redirect()->route('place.oder.index', 'start');
        }

        $currentDate = now()->format('Y-m-d');

        $locationDetails = processLocationDetails();
        $sellings = $locationDetails['sellings'];
        $storages = $locationDetails['storages'];
        $sellingId = $locationDetails['userSellingLocation'];
        $allLocationIds = $sellings->pluck('location_id')->merge($storages->pluck('location_id'))->unique()->values();

        // ✅ Shared counts
        $damageItemsCount = $this->getDamageItemsCount($allLocationIds, $currentDate);
        $itemsMissingPriceCount = $this->getItemsMissingPriceCount($sellings, $sellingId);
        $expiredItemsCount = $this->getExpiredItemsCount($allLocationIds, $sellingId);
        $minimumLevelProducts = $this->getMinimumLevelProducts($allLocationIds, $sellingId);

        $minimumLevelProductsCount = $minimumLevelProducts['count'];

        // 📌 Call your helper functions
        $topMovingProducts = $this->getTopMovingProducts($sellingId);
        $customerDebts = $this->getCustomerDebts($sellingId);
        $supplierDebts = $this->getSupplierDebts($sellingId);
        $requisitionCount = $this->requisitionCount($sellingId);
        $eligibleCustomers = $this->customersWithRoyaltyDiscount($sellingId);

        // Count of eligible customers
        $eligibleCustomersCount = count($eligibleCustomers);

        // 📌 Break them out for the view
        $topMovingProductsCount = $topMovingProducts['count'];
        $topMovingProducts = $topMovingProducts['items'];

        $customerDebtsCount = $customerDebts['count'];
        $customerDebts = $customerDebts['items'];

        $supplierDebtsCount = $supplierDebts['count'];
        $supplierDebts = $supplierDebts['items'];

        [$totalSales, $totalPurchases, $totalExpenses, $totalDue] = $this->getTotals($sellingId, $sellings, $currentDate);

        return view('home', compact(
            'itemsMissingPriceCount',
            'expiredItemsCount',
            'damageItemsCount',
            'minimumLevelProductsCount',
            'totalSales',
            'totalDue',
            'totalPurchases',
            'totalExpenses',
            'topMovingProducts',
            'topMovingProductsCount',
            'customerDebts',
            'customerDebtsCount',
            'supplierDebts',
            'supplierDebtsCount',
            'requisitionCount',
            'sellingId',
            'eligibleCustomers',
            'eligibleCustomersCount'
        ));
    }

    public function royalty()
    {
        $locationDetails = processLocationDetails();
        $sellingId = $locationDetails['userSellingLocation'];

        $result = $this->customersWithRoyaltyDiscount($sellingId);

        // Count of eligible customers
        $eligibleCustomersCount = count($result);

        return view('royalty-report', compact('result', 'eligibleCustomersCount'));
    }

    private function getDamageItemsCount($locationIds, $currentDate)
    {
        return Damage::whereIn('location_id', $locationIds)
            ->whereDate('date', $currentDate)
            ->count();
    }

    private function getItemsMissingPriceCount($sellings, $sellingId)
    {
        $sellingsIds = $sellings->pluck('location_id');

        $missingPricesQuery = Product::whereDoesntHave('unit')
            ->orWhereHas('unit', function ($query) {
                $query->whereNull('selling');
            });

        $missingBuyingPricesQuery = Product::where(function ($query) {
            $query->whereNull('buying')->orWhere('buying', 0);
        });

        if ($sellingId !== 'all') {
            $missingPricesQuery->whereIn('location_id', $sellingsIds);
            $missingBuyingPricesQuery->whereIn('location_id', $sellingsIds);
        }

        $missingPricesCount = $missingPricesQuery->count();
        $missingBuyingPricesCount = $missingBuyingPricesQuery->count();

        return $missingPricesCount + $missingBuyingPricesCount;
    }

    private function getExpiredItemsCount($locationIds, $sellingId)
    {
        $today = now()->format('Y-m-d');
        $aboutToExpireDays = 3;

        $expiredQuery = Stock::whereNotNull('qty')
            ->where('qty', '>', 0)
            ->whereNotNull('exp_date')
            ->whereDate('exp_date', '<=', $today);

        $aboutToExpireQuery = Stock::whereNotNull('qty')
            ->where('qty', '>', 0)
            ->whereNotNull('exp_date')
            ->whereDate('exp_date', '>', $today)
            ->whereDate('exp_date', '<=', now()->addDays($aboutToExpireDays)->format('Y-m-d'));

        if ($sellingId !== 'all') {
            $expiredQuery->whereIn('location_id', $locationIds);
            $aboutToExpireQuery->whereIn('location_id', $locationIds);
        }

        return $expiredQuery->count() + $aboutToExpireQuery->count();
    }

    private function getMinimumLevelProducts($allLocationIds, $sellingId)
    {
        $stockQuery = Stock::whereNotNull('qty')
            ->whereColumn('qty', '<=', 'min_qty'); // Correct comparison!

        if ($sellingId !== 'all') {
            $stockQuery->whereIn('location_id', $allLocationIds);
        }

        // 📌 Get the collection
        $items = $stockQuery->get();

        // 📌 Return both count and collection
        return [
            'count' => $items->count(),
            'items' => $items,
        ];
    }

    private function customersWithRoyaltyDiscount($sellingId)
    {
        $customersQuery = Customer::query();

        if ($sellingId !== 'all') {
            $customersQuery->where('selling_id', $sellingId);
        }

        $customers = $customersQuery->get();
        $result = [];
        $eligibleCustomersCount = 0;

        foreach ($customers as $customer) {
            // Get all sales for this customer (filtered by selling_id if needed)
            $customerSalesQuery = Sale::where('customer_id', $customer->id);
            if ($sellingId !== 'all') {
                $customerSalesQuery->where('selling_id', $sellingId);
            }
            $totalSales = $customerSalesQuery->count();

            // Check if customer has at least 6 sales
            if ($totalSales >= 6) {
                $eligibleCustomersCount++;

                $profile = BusinessProfile::where('selling_id', $customer->selling_id)->first();

                if ($profile && $profile->royalty_enabled) {
                    // Get the last 6 sales regardless of total
                    $lastSixSales = $customerSalesQuery->latest()->take(6)->get();
                    $totalAmount = $lastSixSales->sum('net');
                    $discountAmount = ($totalAmount * $profile->royalty_discount) / 100;

                    $result[] = [
                        'customer_id' => $customer->id,
                        'customer_name' => $customer->name,
                        'total_sales' => $totalSales,
                        'total_amount_last_six' => $totalAmount,
                        'discount_amount' => $discountAmount,
                        'royalty_discount_percent' => $profile->royalty_discount,
                        'eligible_customers_count' => $eligibleCustomersCount,
                    ];
                }
            }
        }

        return $result;
    }

    private function getTotals($sellingId, $sellings, $currentDate)
    {
        if ($sellingId === 'all') {
            $totalPurchases = Purchase::whereDate('date', $currentDate)->sum('net');
            $totalSales = Sale::whereDate('date', $currentDate)->sum('net');
            $totalDue = Sale::whereDate('date', $currentDate)->sum('due');
            $totalExpenses = PostExpense::whereDate('date', $currentDate)->sum('sum');
        } else {
            $totalPurchases = Purchase::whereIn('selling_id', $sellings->pluck('id'))
                ->where('selling_id', $sellingId)
                ->whereDate('date', $currentDate)
                ->sum('net');

            $totalSales = Sale::whereIn('selling_id', $sellings->pluck('id'))
                ->where('selling_id', $sellingId)
                ->whereDate('date', $currentDate)
                ->sum('net');

            $totalDue = Sale::whereIn('selling_id', $sellings->pluck('id'))
                ->where('selling_id', $sellingId)
                ->whereDate('date', $currentDate)
                ->sum('due');

            $totalExpenses = PostExpense::whereIn('selling_id', $sellings->pluck('id'))
                ->where('selling_id', $sellingId)
                ->whereDate('date', $currentDate)
                ->sum('sum');
        }

        return [$totalSales, $totalPurchases, $totalExpenses, $totalDue];
    }

    private function getTopMovingProducts($sellingId)
    {
        $query = DB::table('sold_items')
            ->join('sales', 'sold_items.sale_id', '=', 'sales.id')
            ->join('products', 'sold_items.product_id', '=', 'products.id')
            ->join('units', 'sold_items.unit_id', '=', 'units.id')
            ->select(
                'products.product as name',
                'units.name as unit',
                DB::raw('SUM(sold_items.qty) as total_qty'),
                DB::raw('SUM(sold_items.amount) as total_amount'),
                DB::raw('MAX(sold_items.price) as price')
            );

        if ($sellingId !== 'all') {
            $query->where('sales.selling_id', $sellingId);
        }

        $grouped = $query->groupBy('sold_items.product_id', 'products.product', 'units.name');

        return [
            'count' => $grouped->count(),
            'items' => $grouped->orderByDesc('total_qty')->limit(10)->get(),
        ];
    }

    private function getCustomerDebts($sellingId)
    {
        $latestCustomerDebts = Debts::select('customer_id', DB::raw('MAX(created_at) as latest_entry'))
            ->groupBy('customer_id');

        if ($sellingId !== 'all') {
            $latestCustomerDebts->where('selling_id', $sellingId);
        }

        $query = Debts::joinSub($latestCustomerDebts, 'latest_debts', function ($join) {
            $join->on('debts.customer_id', '=', 'latest_debts.customer_id')
                ->on('debts.created_at', '=', 'latest_debts.latest_entry');
        })->orderBy('debts.due', 'desc');

        return [
            'count' => $query->count(),
            'items' => $query->limit(10)->get(),
        ];
    }

    private function requisitionCount($sellingId)
    {
        $requsitionQuery = Requisition::where('status', 'pending');

        if ($sellingId !== 'all') {
            $requsitionQuery->where('selling_id', $sellingId);
        }

        return $requsitionQuery->count();
    }

    private function getSupplierDebts($sellingId)
    {
        $latestSupplierDebts = SupplierDebts::select('supplier_id', DB::raw('MAX(created_at) as latest_entry'))
            ->groupBy('supplier_id');

        if ($sellingId !== 'all') {
            $latestSupplierDebts->where('selling_id', $sellingId);
        }

        $query = SupplierDebts::joinSub($latestSupplierDebts, 'latest_debts', function ($join) {
            $join->on('supplier_debts.supplier_id', '=', 'latest_debts.supplier_id')
                ->on('supplier_debts.created_at', '=', 'latest_debts.latest_entry');
        })->orderBy('supplier_debts.due', 'desc');

        return [
            'count' => $query->count(),
            'items' => $query->limit(10)->get(),
        ];
    }

    public function paymentReminder()
    {
        try {
            return view("payment");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function ipcheck()
    {
        try {
            return view("ipcheck");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function madeniyaleowateja()
    {
        try {
            $madeni = Sale::where("customer", "!=", "")
                ->where("status", "credit")
                ->where("duedate", date('Y-m-d'))
                ->get();

            return view("madeni-ya-leo-wateja", compact("madeni"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function madeniyamawaiter()
    {
        try {
            $madeni = Sale::where("waiter", "!=", "")
                ->where("status", "credit")
                ->get();

            return view("madeni-ya-mawaiter", compact("madeni"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function badDebts()
    {
        try {
            $madeni = Sale::where("status", "credit")
                ->where("duedate", "<", date('Y-m-d'))
                ->get();

            return view("bad-debts", compact("madeni"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function expiredProducts()
    {
        try {
            // Get location-related data
            $locationDetails = processLocationDetails();
            $sellings = $locationDetails['sellings'];
            $storages = $locationDetails['storages'];
            $sellingId = $locationDetails['userSellingLocation'];

            $allLocationIds = $sellings->pluck('location_id')
                ->merge($storages->pluck('location_id'))
                ->unique()
                ->values();

            $today = now()->format('Y-m-d');
            $aboutToExpireDays = 30;

            // ✅ Base query
            $expiredQuery = Stock::whereNotNull('qty')
                ->where('qty', '>', 0)
                ->whereNotNull('exp_date')
                ->whereDate('exp_date', '<=', $today)
                ->with('product.location', 'unit');

            $aboutToExpireQuery = Stock::whereNotNull('qty')
                ->where('qty', '>', 0)
                ->whereNotNull('exp_date')
                ->whereDate('exp_date', '>', $today)
                ->whereDate('exp_date', '<=', now()->addDays($aboutToExpireDays)->format('Y-m-d'))
                ->with('product.location', 'unit');

            // ✅ Apply location filter ONLY if not 'all'
            if ($sellingId != 'all') {
                $expiredQuery->whereIn('location_id', $allLocationIds);
                $aboutToExpireQuery->whereIn('location_id', $allLocationIds);
            }

            $expiredProducts = $expiredQuery->get();
            $aboutToExpireProducts = $aboutToExpireQuery->get();

            return view('expired-products', compact('expiredProducts', 'aboutToExpireProducts'));
        } catch (Exception $e) {
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function minimumLevel()
    {
        try {
            $locationDetails = processLocationDetails();
            $sellings = $locationDetails['sellings'];
            $storages = $locationDetails['storages'];
            $sellingId = $locationDetails['userSellingLocation'];
            $allLocationIds = $sellings->pluck('location_id')->merge($storages->pluck('location_id'))->unique()->values();

            $minimumLevelProducts = $this->getMinimumLevelProducts($allLocationIds, $sellingId);

            $minimumLevelProducts = $minimumLevelProducts['items'];

            return view('minimum-level-product', compact('minimumLevelProducts'));
        } catch (Exception $e) {
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function itemsMissingPrices($sellingId)
    {
        $locationIds = [];

        if ($sellingId !== 'all') {
            $locationIds = Selling::where('id', $sellingId)
                ->pluck('location_id')
                ->toArray();
        }

        // Common query builder for products missing selling prices
        $missingSellingPricesQuery = Product::where(function ($query) {
            $query->whereDoesntHave('unit')
                ->orWhereHas('unit', function ($q) {
                    $q->whereNull('selling');
                });
        });

        if (!empty($locationIds)) {
            $missingSellingPricesQuery->whereIn('location_id', $locationIds);
        }

        $missingSellingPrices = $missingSellingPricesQuery
            ->with(['unit' => function ($query) {
                $query->whereNull('selling');
            }])
            ->get();

        // Products missing buying prices
        $missingBuyingPricesQuery = Product::where(function ($query) {
            $query->whereNull('buying')->orWhere('buying', 0);
        });

        if (!empty($locationIds)) {
            $missingBuyingPricesQuery->whereIn('location_id', $locationIds);
        }

        $missingBuyingPrices = $missingBuyingPricesQuery->get();

        return view("missing-prices", compact("missingBuyingPrices", "missingSellingPrices"));
    }

    public function todaysPendingPurchaseView()
    {
        try {
            $madeni = Purchase::where("status", "credit")
                ->where("duedate", "=", date('Y-m-d'))
                ->get();

            $totaldebt = Purchase::where("status", "credit")
                ->where("duedate", "=", date('Y-m-d'))
                ->sum("due");

            return view("today-pending-purchases", compact("madeni", "totaldebt"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function passedPendingPurchaseView()
    {
        try {
            $madeni = Purchase::where("status", "credit")
                ->where("duedate", "<", date('Y-m-d'))
                ->get();

            $totaldebt = Purchase::where("status", "credit")
                ->where("duedate", "<", date('Y-m-d'))
                ->sum("due");

            return view("passed-pending-purchases", compact("madeni", "totaldebt"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
}
