<?php

namespace App\Http\Controllers;

use App\Models\Account;
use App\Models\Creditor;
use App\Models\Loan;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class CreditorController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        # Get location details
        $locationDetails = processLocationDetails();
        $sellingCount = $locationDetails['sellingCount'];
        $sellings = $locationDetails['sellings'];
        $accounts = $locationDetails['accounts'];
        $creditors = $locationDetails['creditors'];
        $loans = $locationDetails['loans'];

        return view('creditor', compact(
            "creditors",
            "sellingCount",
            "sellings",
            'loans'
        ));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        DB::beginTransaction();
        try {
            $creditor = new Creditor();
            $creditor->name = $request->input('name');
            $creditor->address = $request->input('address');
            $creditor->phone = $request->input('phone');
            $creditor->location_id = $request->input('location');
            $creditor->save();

            DB::commit();

            return redirect()->back()->with('success', 'Successfully added new creditor.');
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['error' => 'An error occurred while registering creditor.']);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        try {
            $creditor = Creditor::find($id);
            $creditor->name = $request->input('name');
            $creditor->address = $request->input('address');
            $creditor->phone = $request->input('phone');
            $creditor->location_id = $request->input('location');
            $creditor->update();

            DB::commit();

            return redirect()->back()->with('success', 'Successfully updated.');
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['error' => 'An error occurred while updating creditor.']);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        DB::beginTransaction();

        try {
            Creditor::destroy($id);
            DB::commit();
            return redirect()->back()->with("success", "Successfully deleted!!");
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'An error occurred while deleting creditor']);
        }
    }

    public function recievedLoanIndex()
    {
        # Get location details
        $locationDetails = processLocationDetails();
        $sellingCount = $locationDetails['sellingCount'];
        $creditors = $locationDetails['creditors'];
        $accounts = $locationDetails['accounts'];

        $loans = Loan::with('location', 'user', 'account')->get();

        return view('recieved-loan', compact("creditors", "loans", "accounts"));
    }


    public function getNextInstallment($issueDate, $returnDate, $numInstallments)
    {
        // Convert dates to Carbon instances for easy date manipulation
        $issueDate = Carbon::parse($issueDate);
        $returnDate = Carbon::parse($returnDate);

        // Calculate total duration of loan in days
        $totalDuration = $returnDate->diffInDays($issueDate);

        // Determine the duration of each installment period in days
        $installmentDuration = $totalDuration / $numInstallments;

        // Calculate current number of installments paid
        $currentDate = Carbon::now();
        $currentDuration = $currentDate->diffInDays($issueDate);
        $currentInstallments = floor($currentDuration / $installmentDuration);

        // Calculate next installment date
        $nextInstallmentDate = $issueDate->copy()->addDays(($currentInstallments + 1) * $installmentDuration);

        return [
            'nextInstallmentDate' => $nextInstallmentDate->toDateString(),
        ];
    }

    public function recievedLoanStore(Request $request)
    {
        DB::beginTransaction();

        try {

            $details = $this->getNextInstallment($request->date, $request->rdate, $request->installment);

            Loan::create([
                'date' => $request->date,
                'creditor_id' => $request->creditor,
                'principle' => $request->principle,
                'account_id' => $request->account,
                'rdate' => $request->rdate,
                'purpose' => $request->purpose,
                'intrest' => $request->intrest,
                'tamount' => $request->tamount,
                'installment' => $request->installment,
                'iamount' => $request->iamount,
                'paid' => 0,
                'status' => 1,
                'next_installment_date' => $details["nextInstallmentDate"],
                'next_installment_amount' => $request->iamount,
                'location_id' => auth()->user()->selling_location_id,
                'user_id' => auth()->user()->id,
            ]);

            $account = Account::find($request->account);

            $transaction = new Transaction();
            $transaction->date = $request->date;
            $transaction->user_id = auth()->user()->id;
            $transaction->reason = "Loan recieved";
            $transaction->status = "In";
            $transaction->amount = $request->principle;
            $transaction->before = $account->balance;
            $transaction->after = $account->balance + $request->principle;
            $transaction->account_id = $request->account;
            $transaction->location_id = auth()->user()->selling_location_id;
            $transaction->save();

            $account->balance = $account->balance + $request->principle;
            $account->update();

            DB::commit();

            return redirect()->route("recieved.loan.index")->with("success", "Successfully added loan");
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'An error occured while adding loan'], 422);
        }
    }

    public function updateLoan(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            $loan = Loan::find($id);

            $loan->update([
                'date' => $request->date,
                'creditor_id' => $request->creditor,
                'principle' => $request->principle,
                'rdate' => $request->rdate,
                'purpose' => $request->purpose,
                'interest' => $request->intrest,
                'tamount' => $request->tamount,
                'installment' => $request->installment,
                'iamount' => $request->iamount,
                'paid' => $request->paid,
            ]);

            DB::commit();

            return redirect()->back()->with("success", "Successfully updated loan");
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with("error", "An error occurred while updating loan");
        }
    }

    public function destroyLoan($id)
    {
        DB::beginTransaction();

        try {
            Loan::destroy($id);

            DB::commit();

            return redirect()->back()->with("success", "Successfully deleted!!");
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'An error occurred while deleting loan'], 422);
        }
    }

    public function loanPayment(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            $loan = Loan::find($id);
            $loan->paid += $request->paid;
            $loan->save();

            if ($loan->paid >= $loan->tamount) {
                $loan->update(['status' => 2]);
            } else {
                $details = $this->getNextInstallment($loan->next_installment_date, $loan->rdate, $loan->installment);
                $loan->update(['next_installment_date' => $details["nextInstallmentDate"]]);
            }

            DB::commit();

            return redirect()->back()->with("success", "Successfully paid loan");
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with("error", "An error occurred while processing loan payment");
        }
    }
}
