<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Models\BusinessProfile;
use App\Models\Location;
use App\Models\Selling;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class BusinessProfileController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        try {
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $userSellingLocation = $locationDetails['userSellingLocation'];

            $profileData = collect();

            if ($sellingCount == 1) {
                $profileData = BusinessProfile::where('selling_id', $userSellingLocation)->get();
            }

            if ($request->has('location')) {
                $profileData = BusinessProfile::where('selling_id', $request->location)->get();
            }

            return view('business-profile', compact(
                "profileData",
                "sellings"
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        //try {
        $profile = BusinessProfile::find($id);

        if (!$profile) {
            DB::rollBack();
            return back()->with("error", "Business profile not found!!");
        }

        // Validate the base data + royalty fields
        $request->validate([
            'name' => 'required|string',
            'address' => 'required|string',
            'phone' => 'required|string',
            'vat' => 'required|string',
            'receipt' => 'required|string',
            'royalty' => 'required|boolean',
            'royalty_discount' => 'nullable|numeric|min:1|max:100',
            'batch_no' => 'required|boolean',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'email' => 'nullable|email',
            // 'website' => 'nullable|url',
            'tin' => 'nullable|string',
            'vrn' => 'nullable|string',
            'slogan' => 'nullable|string',
            'qrcode' => 'required|boolean',
        ]);

        // Handle logo upload
        if ($request->hasFile('logo')) {
            $image = $request->file('logo');
            $fileName = uniqid('logo_') . '.' . $image->getClientOriginalExtension();

            // Delete old logo if exists
            if ($profile->logo && Storage::disk('public')->exists('logos/' . $profile->logo)) {
                Storage::disk('public')->delete('logos/' . $profile->logo);
            }

            // Store new logo
            $image->storeAs('logos', $fileName, 'public');
            $profile->logo = $fileName;
        }

        // Get location id from selling
        $locationId = Selling::find($profile->selling_id)?->location_id;
        if (!$locationId) {
            DB::rollBack();
            return back()->with("error", "Location not found for this business profile!!");
        }

        // Update related location name
        Location::find($locationId)?->update([
            'name' => ucwords($request->name)
        ]);

        // Update the profile fields
        $profile->address = ucwords($request->address);
        $profile->phone = $request->phone;
        $profile->email = $request->email;
        $profile->website = $request->website;
        $profile->tin = $request->tin;
        $profile->vrn = $request->vrn;
        $profile->slogan = ucfirst($request->slogan);
        $profile->vat = $request->vat;
        $profile->receipt = $request->receipt;

        // 🆕 Royalty rule
        $profile->royalty_enabled = $request->royalty;
        $profile->royalty_discount = $request->royalty == 1 ? $request->royalty_discount : null;

        // Batch number
        $profile->batch_no = $request->batch_no;

        // QR Code on receipt
        $profile->qrcode = $request->qrcode;

        $profile->save();

        DB::commit();
        return back()->with("success", "Business profile successfully updated!!");
        // } catch (\Exception $e) {
        //     DB::rollBack();
        //     return back()->with("error", "Something went wrong while updating the business profile! " . $e->getMessage());
        // }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }

    public function pricing()
    {
        return view('pricing');
    }
}
