<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Account;
use App\Models\Selling;
use App\Models\Transaction;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class AccountController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $accounts = $locationDetails['accounts'];
            $mains = $locationDetails['mains'];

            return view('account', compact(
                "accounts",
                "sellingCount",
                "mains",
                "sellings",
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function accountBalance(Request $request)
    {
        try {
            // Initialize variables
            $accountId = $request->account;
            $date = $request->date;
            $balance = null;

            // Calculate the total amount in
            $totalIn = Transaction::where('account_id', $accountId)
                ->where('date', $date)
                ->where('status', 'In')
                ->sum('amount');

            // Calculate the total amount out
            $totalOut = Transaction::where('account_id', $accountId)
                ->where('date', $date)
                ->where('status', 'Out')
                ->sum('amount');

            // Calculate the account balance
            $balance = $totalIn - $totalOut;

            // Return the account balance as a JSON response
            return response()->json(['balance' => $balance, 'account' => Account::find($accountId)]);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function AccountBalancePerDuration(Request $request)
    {
        try {
            // Get 'type', 'start', 'end', and the current date from the request
            $type = $request->get('type');
            $start = $request->get('start');
            $end = $request->get('end');
            $currentDate = date('Y-m-d');

            // Initialize an empty array to store unique account data
            $accountData = [];

            // Check if the type is 'today'
            if ($type == 'today') {
                // Retrieve the latest transactions with associated accounts and get unique accounts
                $transactions = Transaction::where('date', $currentDate)->get();

                // Loop through the unique latest transactions
                foreach ($transactions as $transaction) {
                    // Get the associated account and related information
                    $account = $transaction->account;
                    $accountID = $account->id;
                    $amount = $transaction->amount;

                    // Check if the account ID is not already in the accountData array
                    if (!isset($accountData[$accountID])) {
                        // Add a new entry to accountData with 'account_id' and 'amount'
                        $accountData[$accountID] = [
                            'account_id' => $accountID,
                            'amount' => $amount,
                        ];

                        continue;
                    }

                    // Check the status of the transaction
                    $status = $transaction->status;

                    // Adjust the amount based on the transaction status
                    if ($status == 'In') {
                        $accountData[$accountID]['amount'] += $transaction->amount;
                    } elseif ($status == 'Out') {
                        $accountData[$accountID]['amount'] -= $transaction->amount;
                    }
                }
            } elseif ($type == 'set date') {
                // Retrieve the latest transactions with associated accounts and get unique accounts
                $transactions = Transaction::where('date', $start)->get();

                // Loop through the unique latest transactions
                foreach ($transactions as $transaction) {
                    // Get the associated account and related information
                    $account = $transaction->account;
                    $accountID = $account->id;
                    $amount = $transaction->amount;

                    // Check if the account ID is not already in the accountData array
                    if (!isset($accountData[$accountID])) {
                        // Add a new entry to accountData with 'account_id' and 'amount'
                        $accountData[$accountID] = [
                            'account_id' => $accountID,
                            'amount' => $amount,
                        ];

                        continue;
                    }

                    // Check the status of the transaction
                    $status = $transaction->status;

                    // Adjust the amount based on the transaction status
                    if ($status == 'In') {
                        $accountData[$accountID]['amount'] += $transaction->amount;
                    } elseif ($status == 'Out') {
                        $accountData[$accountID]['amount'] -= $transaction->amount;
                    }
                }
            } elseif ($type == 'set duration') {
                // Retrieve the latest transactions with associated accounts and get unique accounts
                $transactions = Transaction::whereBetween('date', [$start, $end])->get();

                // Loop through the unique latest transactions
                foreach ($transactions as $transaction) {
                    // Get the associated account and related information
                    $account = $transaction->account;
                    $accountID = $account->id;
                    $amount = $transaction->amount;

                    // Check if the account ID is not already in the accountData array
                    if (!isset($accountData[$accountID])) {
                        // Add a new entry to accountData with 'account_id' and 'amount'
                        $accountData[$accountID] = [
                            'account_id' => $accountID,
                            'amount' => $amount,
                        ];

                        continue;
                    }

                    // Check the status of the transaction
                    $status = $transaction->status;

                    // Adjust the amount based on the transaction status
                    if ($status == 'In') {
                        $accountData[$accountID]['amount'] += $transaction->amount;
                    } elseif ($status == 'Out') {
                        $accountData[$accountID]['amount'] -= $transaction->amount;
                    }
                }
            }

            // Format the 'amount' using number_format
            foreach ($accountData as &$account) {
                $account['amount'] = number_format($account['amount']);
            }

            // Return the account balance as a JSON response
            return response()->json($accountData);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            // Validate the request data
            $validatedData = $request->validate([
                'name'    => 'required',
                'type'    => 'required',
                'number'  => 'required',
                'balance' => 'required|numeric',
                'main'    => 'required',
                'selling' => 'required',
            ]);

            // Create the Account record
            $account = Account::create([
                'name'       => ucwords($validatedData['name']),
                'type'       => ucwords($validatedData['type']),
                'number'     => $validatedData['number'],
                'balance'    => $validatedData['balance'],
                'selling_id' => $validatedData['selling'],
            ]);

            // Record the initial deposit using the helper
            recordTransaction(
                date('Y-m-d'),              // date
                'Deposit',                  // reason
                'In',                       // status
                $validatedData['balance'],  // amount
                $account->id,               // account_id
                auth()->id(),               // user_id
            );

            return back()->with('success', 'Successfully added!');
        } catch (Exception $e) {
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }


    public function cashDeposite(Request $request)
    {
        try {
            // Validate the request data
            $validatedData = $request->validate([
                'account' => 'required',
                'amount'  => 'required|numeric',
                'date'    => 'required|date',
            ]);

            // Record the deposit transaction (this also updates account balance)
            recordTransaction(
                $validatedData['date'],     // date
                'Deposit',                  // reason
                'In',                       // status
                $validatedData['amount'],   // amount
                $validatedData['account'],  // account_id
                auth()->id()                // user_id
            );

            return back()->with("success", "Successfully Deposited!!");
        } catch (Exception $e) {
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function cashTransfer(Request $request)
    {
        return DB::transaction(function () use ($request) {
            $accounts = Account::find([$request->from, $request->to]);

            if (count($accounts) !== 2) {
                return back()->with("error", "One or more accounts not found. Transfer failed.");
            }

            [$from, $to] = $accounts;

            // Record transaction for "from" account (money going out)
            recordTransaction(
                $request->date,
                'Transfer',
                'Out',
                $request->amount,
                $from->id,
                auth()->id()
            );

            // Record transaction for "to" account (money coming in)
            recordTransaction(
                $request->date,
                'Transfer',
                'In',
                $request->amount,
                $to->id,
                auth()->id()
            );

            return back()->with("success", "Successfully Transferred.");
        });
    }

    public function cashWithdraw(Request $request)
    {
        $validatedData = $request->validate([
            'account' => 'required',
            'amount' => 'required|numeric',
            'date' => 'required|date',
        ]);

        try {
            DB::beginTransaction();

            // Record the withdraw transaction (this also updates account balance)
            recordTransaction(
                $validatedData['date'],
                'Withdraw',
                'Out',
                $validatedData['amount'],
                $validatedData['account'],
                auth()->id()
            );

            DB::commit();

            return back()->with("success", "Successfully Withdrawn.");
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with("error", "Withdrawal failed. Please try again.");
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        // ✅ Validate the incoming request
        $validatedData = $request->validate([
            'name' => 'required',
            'type' => 'required',
            'number' => 'required',
            'balance' => 'required|numeric',
            'main' => 'required',
            'selling' => 'required',
        ]);

        // Find the account by ID
        $account = Account::find($id);

        // Begin database transaction to ensure consistency
        DB::beginTransaction();

        // try {
        // Store the old balance for comparison
        $oldBalance = $account->balance;
        $newBalance = $validatedData['balance'];

        // Determine transaction status & amount based on balance change
        if ($newBalance > $oldBalance) {
            // Deposit case
            $status = "In";
            $amount = $newBalance - $oldBalance;
        } elseif ($newBalance < $oldBalance) {
            // Withdrawal case
            $status = "Out";
            $amount = $oldBalance - $newBalance;
        } else {
            // No balance change → skip transaction recording
            $status = null;
            $amount = 0;
        }

        // ✅ Record the transaction if there was a change in balance
        if ($status) {
            recordTransaction(
                date('Y-m-d'),             // date
                'Updated',                  // reason
                $status,                    // status (In/Out)
                $amount,                    // amount changed
                $account->id,               // account_id
                auth()->id(),               // user_id
            );
        }

        // ✅ Update account details
        $account->update([
            'name'       => ucwords($validatedData['name']),
            'type'       => ucwords($validatedData['type']),
            'number'     => $validatedData['number'],
            'balance'    => $newBalance,
            'selling_id' => $validatedData['selling'],
        ]);

        // Commit all DB changes
        DB::commit();

        return back()->with("success", "Successfully updated.");
        // } catch (\Exception $e) {
        //     // Rollback changes if something went wrong
        //     DB::rollBack();
        //     return back()->with("error", "Update failed. Please try again.");
        // }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        // ✅ Find the account
        $account = Account::find($id);

        // Start DB transaction to ensure all-or-nothing execution
        DB::beginTransaction();

        try {
            // ✅ Record the transaction for deleting the account
            recordTransaction(
                date('Y-m-d'),         // date
                auth()->id(),          // user_id
                'Deleted',             // reason
                'Out',                 // status
                $account->balance,     // amount
                $account->balance,     // before
                0,                     // after
                $account->id,          // account_id
                $account->selling_id   // selling_id
            );

            // ✅ Soft delete the account
            $account->delete();

            // Commit DB changes
            DB::commit();

            return back()->with("success", "Successfully deleted.");
        } catch (\Exception $e) {
            // Rollback if something fails
            DB::rollBack();
            return back()->with("error", "Deletion failed. Please try again.");
        }
    }


    public function getLocationAccounts(Request $request)
    {
        try {
            $locationId = $request->get('locationId');
            $sellingId = Selling::where('location_id', $locationId)->value('id');

            if (!$sellingId) {
                return response()->json(['error' => 'No selling found for this location'], 404);
            }

            $accounts = Account::where("selling_id", $sellingId)->get();
            return response()->json($accounts);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function bulkDelete(Request $request)
    {
        DB::beginTransaction();

        try {
            $ids = $request->ids;

            foreach ($ids as $id) {
                // Find account
                $account = Account::find($id);

                if ($account) {
                    // Record the transaction before deletion
                    recordTransaction(
                        date('Y-m-d'),         // date
                        auth()->id(),          // user_id
                        'Bulk Deleted',        // reason
                        'Out',                 // status
                        $account->balance,     // amount
                        $account->balance,     // before
                        0,                     // after
                        $account->id,          // account_id
                        $account->selling_id   // selling_id
                    );

                    // Soft delete the account
                    $account->delete();
                }
            }

            DB::commit();
            return response()->json(['success' => true, 'message' => 'Accounts deleted successfully.']);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
}
